/****************************************************************************

    PROGRAM: homeadm.c

    PURPOSE: Windows client to RPC HomeDrive Server

    Author:
        Joev Dubach


    Purpose:
        HOMEADM is the companion administrative app to HOMEDRV.DLL, a
        File Manager extension DLL.  It allows the database to be
        edited.  For more information, see the source to HOMEDRV.DLL,
        homedrv.c.

****************************************************************************/

//
// Inclusions
//

#include <windows.h>                // required for all Windows applications
#include <stdlib.h>
#include <windows.h>
#include <string.h>
#include <stdio.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <ctype.h>
#include <rpc.h>                    // RPC data structures and APIs
#include <rpcerr.h>
#include "home.h"                   // header file generated by MIDL compiler
#include "homeadm.h"                // specific to this program
#include "..\homedir.h"             // Server/client error codes

//
// Global variables
//

HANDLE hInst;                       // current instance
WPARAM SelectedMenu;                // currently selected menu item
BOOL ValidBinding=FALSE;

//
// Function prototypes
//

int PASCAL
WinMain(
    HANDLE hInstance,
    HANDLE hPrevInstance,
    LPSTR lpCmdLine,
    int nCmdShow
    );
BOOL
InitApplication(
    HANDLE hInstance
    );
BOOL
InitInstance(
    HANDLE hInstance,
    int nCmdShow
    );
long FAR PASCAL
MainWndProc(
    HWND hWnd,
    UINT message,
    WPARAM wParam,
    LPARAM lParam
    );
BOOL FAR PASCAL
MenuADG(
    HWND hDlg,
    unsigned message,
    WORD wParam,
    LONG lParam
    );
void
CheckRpcStatus(
    char *ProcName,
    RPC_STATUS status
    );
void
CheckMyStatus(
    char *ProcName,
    MY_STATUS status
    );
BOOL FAR PASCAL
AboutDlgProc (
    HWND hDlg,
    unsigned uMessage,
    WORD wParam,
    LONG lParam
    );
BOOL FAR PASCAL
SettingsProc(
    HWND hDlg,
    unsigned message,
    WORD wParam,
    LONG lParam
    );
void
CreateBinding(
    char *Address
    );
void
DestroyBinding(
    void
    );

//
// Main program
//

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point
        for the program.  This function calls the application initialization
        routine, if no other instance of the program is running, and always
        calls the instance initialization routine.  It then executes a message
        retrieval and dispatch loop that is the top-level control structure
        for the remainder of execution.  The loop is terminated when a WM_QUIT
        message is received, at which time this function exits the application
        instance by returning the value passed by PostQuitMessage().

        If this function must abort before entering the message loop, it
        returns the conventional value NULL.

****************************************************************************/

int PASCAL
WinMain(
    HANDLE hInstance,                            // current instance
    HANDLE hPrevInstance,                        // previous instance
    LPSTR lpCmdLine,                             // command line
    int nCmdShow                                 // show-window type (open/icon)
    )
{
    MSG msg;                                 // message

    if (!hPrevInstance)                  // Other instances of app running?
        {
        if (!InitApplication(hInstance)) // Initialize shared things
            {
            return (FALSE);              // Exits if unable to initialize
            }
        }
    // Perform initializations that apply to a specific instance

    if (!InitInstance(hInstance, nCmdShow))
        {
        return (FALSE);
        }

    // Acquire and dispatch messages until a WM_QUIT message is received.

    while (GetMessage(&msg,        // message structure
            NULL,                  // handle of window receiving the message
            NULL,                  // lowest message to examine
            NULL))                 // highest message to examine
        {
        TranslateMessage(&msg);    // Translates virtual key codes
        DispatchMessage(&msg);     // Dispatches message to window
    }
    if (ValidBinding)
        {
        DestroyBinding();
        }

    return (msg.wParam);           // Returns the value from PostQuitMessage
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other
        instances of the application are running.  This function performs
        initialization tasks that can be done once for any number of running
        instances.

        In this case, we initialize a window class by filling out a data
        structure of type WNDCLASS and calling the Windows RegisterClass()
        function.  Since all instances of this application use the same window
        class, we only need to do this when the first instance is initialized.


****************************************************************************/

BOOL
InitApplication(
    HANDLE hInstance                               // current instance
    )
{
    WNDCLASS  wc;
    char pszNetworkAddress[STRINGSIZE]="";

    if(!GetPrivateProfileString("HomeDrive",   // Section of winfile.ini
               "Server",                       // Variable
               "",                             // Default
               pszNetworkAddress,              // Where to put it
               STRINGSIZE,
               "winfile.ini"))
        {
        CreateBinding(pszNetworkAddress);
        ValidBinding=TRUE;
        }
    // Fill in window class structure with parameters that describe the
    // main window.

    wc.style = NULL;                    // Class style(s).
    wc.lpfnWndProc = MainWndProc;       // Function to retrieve messages for
                                        // windows of this class.
    wc.cbClsExtra = 0;                  // No per-class extra data.
    wc.cbWndExtra = 0;                  // No per-window extra data.
    wc.hInstance = hInstance;           // Application that owns the class.
    wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(MAINICON));
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "HOMEDRIVEMENU"; // Name of menu resource in .RC file.
    wc.lpszClassName = "GenericWClass"; // Name used in call to CreateWindow.

    // Register the window class and return success/failure code.

    return (RegisterClass(&wc));

}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of
        this application.  This function performs initialization tasks that
        cannot be shared by multiple instances.

        In this case, we save the instance handle in a static variable and
        create and display the main program window.

****************************************************************************/

BOOL
InitInstance(
    HANDLE          hInstance,          // Current instance identifier.
    int             nCmdShow            // Param for first ShowWindow() call.
    )
{
    HWND            hWnd;               // Main window handle.

    // Save the instance handle in static variable, which will be used in
    // many subsequence calls from this application to Windows.

    hInst = hInstance;

    // Create a main window for this application instance.

    hWnd = CreateWindow(
        "GenericWClass",                // See RegisterClass() call.
        "HomeDrive Administrative Application",// Text for window title bar.
        WS_OVERLAPPEDWINDOW,            // Window style.
        CW_USEDEFAULT,                  // Default horizontal position.
        CW_USEDEFAULT,                  // Default vertical position.
        CW_USEDEFAULT,                  // Default width.
        CW_USEDEFAULT,                  // Default height.
        NULL,                           // Overlapped windows have no parent.
        NULL,                           // Use the window class menu.
        hInstance,                      // This instance owns this window.
        NULL                            // Pointer not needed.
    );

    // If window could not be created, return "failure"

    if (!hWnd)
        {
        return (FALSE);
        }

    // Make the window visible; update its client area; and return "success"

    ShowWindow(hWnd, nCmdShow);  // Show the window
    UpdateWindow(hWnd);          // Sends WM_PAINT message
    return (TRUE);               // Returns the value from PostQuitMessage

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

        WM_COMMAND    - application menu (About dialog box)
        WM_DESTROY    - destroy window

    COMMENTS:

        To process the IDM_ABOUT message, call MakeProcInstance() to get the
        current instance address of the About() function.  Then call Dialog
        box which will create the box according to the information in your
        generic.rc file and turn control over to the About() function.  When
        it returns, free the intance address.

****************************************************************************/

long FAR PASCAL
MainWndProc(
    HWND hWnd,                              // window handle
    UINT message,                           // type of message
    WPARAM wParam,                          // additional information
    LPARAM lParam                           // additional information
    )
{
    FARPROC lpProc;                     // pointer to the chosen function

    switch (message) {
        case WM_COMMAND:                // command from application menu
            SelectedMenu = wParam;
            switch(wParam) {
                case IDM_ADD:
                    lpProc = MakeProcInstance(MenuADG, hInst);
                    DialogBox(hInst,    // current instance
                        "ADDBOX",       // resource to use
                        hWnd,           // parent handle
                        lpProc);        // MenuADG() instance address
                    FreeProcInstance(lpProc);
                    break;

                case IDM_DELETE:
                    lpProc = MakeProcInstance(MenuADG, hInst);
                    DialogBox(hInst,    // current instance
                        "DELETEBOX",    // resource to use
                        hWnd,           // parent handle
                        lpProc);        // MenuADG() instance address
                    FreeProcInstance(lpProc);
                    break;

                case IDM_GET:
                    lpProc = MakeProcInstance(MenuADG, hInst);
                    DialogBox(hInst,    // current instance
                        "GETBOX",       // resource to use
                        hWnd,           // parent handle
                        lpProc);        // MenuADG() instance address
                    FreeProcInstance(lpProc);
                    break;

                case IDM_ABOUT:
                    lpProc = MakeProcInstance(AboutDlgProc, hInst);
                    DialogBox (hInst,
                               "ABOUTBOX",
                               hWnd,
                               lpProc);
                    FreeProcInstance(lpProc);
                    break;

                case IDM_SETTINGS:
                    lpProc = MakeProcInstance(SettingsProc, hInst);
                    DialogBox (hInst,
                               "SETTINGSBOX",
                               hWnd,
                               lpProc);
                    FreeProcInstance(lpProc);
                    break;

                case IDM_EXIT:
                    DestroyWindow(hWnd);
                    break;

                default:
                    return (DefWindowProc(hWnd, message, wParam, lParam));

                }
            FreeProcInstance(lpProc);
            break;

        case WM_DESTROY:                  // message: window being destroyed
            PostQuitMessage(0);
            break;

        default:                          // Passes it on if unproccessed
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: MenuADG(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for the "Add," "Delete," and "Get" dialog
              boxes.  It does this by using the global variable SelectedMenu.
              (I can't change the parameters to MenuADG, as I "call" it in
              the Windows sense: by using MakeProcInstance and DialogBox; thus,
              SelectedMenu is the parameter I cannot pass.

    MESSAGES:

        WM_INITDIALOG - initialize dialog box
        WM_COMMAND    - Input received

    COMMENTS:

        No initialization is needed for this particular dialog box, but TRUE
        must be returned to Windows.

        Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL
MenuADG(
    HWND hDlg,                              // window handle of the dialog box
    unsigned message,                       // type of message
    WORD wParam,                            // message-specific information
    LONG lParam
    )
{
    char name[STRINGSIZE];
    char dir[STRINGSIZE];
    MY_STATUS TempStat;
    int i;
    BOOL SyntaxOK=TRUE;

    switch (message) {
        case WM_INITDIALOG:             // message: initialize dialog box
            if (!ValidBinding)
                {
                MessageBox(hDlg,"You do not have a valid HomeDrive server selected.  Please select Settings from the HomeDrive menu and enter your server name.","Error",MB_OK);
                EndDialog(hDlg, FALSE); // Exits the dialog box
                }
            return (TRUE);

        case WM_COMMAND:                // message: received a command
            switch (wParam)
                {
                case IDOK:
                    GetDlgItemText(hDlg,IDNAME,name,STRINGSIZE);
                    if (SelectedMenu == IDM_ADD)
                        {
                        SyntaxOK=FALSE;
                        GetDlgItemText(hDlg,IDDIR,dir,STRINGSIZE);
                        if ((dir[0]=='\\')&&(dir[1]=='\\'))
                            {
                            for(i=2;(dir[i]!='\\')&&(dir[i]!='\0');i++)
                                {
                                }
                            if (dir[i]=='\\')
                                {
                                SyntaxOK=TRUE;
                                }
                            }
                        }

                    if (SyntaxOK)
                        {
                        RpcTryExcept
                            {
                            switch(SelectedMenu)
                                {
                                case IDM_ADD:
                                    CheckMyStatus(
                                        "RPC Add call",
                                        TempStat = RpcHomesvrAdd(name,dir)
                                        );
                                    break;
                                case IDM_DELETE:
                                    CheckMyStatus(
                                        "RPC Delete call",
                                        TempStat = RpcHomesvrDel(name)
                                        );
                                    break;
                                case IDM_GET:
                                    CheckMyStatus(
                                        "RPC Get call",
                                        TempStat = RpcHomesvrGet(name,dir)
                                        );
                                    if (!TempStat)
                                        {
                                        MessageBox(hDlg,
                                                   dir,
                                                   "The directory is:",
                                                   MB_OK);
                                        }
                                    break;
                                default:
                                    MessageBox(hDlg,"Invalid Menu ID",
                                               "Fatal Error",MB_OK);
                                    exit(1);
                                }
                            }
                        RpcExcept(1)
                            {
                            CheckRpcStatus(
                                "Runtime library reported an exception",
                                RpcExceptionCode()
                                );
                            }
                        RpcEndExcept
                        }
                    else
                        {
                        MessageBox(hDlg,"Invalid share syntax","Error",MB_OK);
                        TempStat=TRUE;
                        }
                    if (!TempStat)
                        {
                        EndDialog(hDlg, TRUE);
                        }
                    return (TRUE);

                case IDCANCEL:
                    EndDialog(hDlg, FALSE);        // Exits the dialog box
                    return (TRUE);

                default:
                    DefWindowProc(hDlg, message, wParam, lParam);
                }
            break;
    }
    return (FALSE);                           // Didn't process a message
}

/****************************************************************************

    FUNCTION: SettingsProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for the "Settings" dialog box.

    MESSAGES:

        WM_INITDIALOG - initialize dialog box
        WM_COMMAND    - Input received

****************************************************************************/

BOOL FAR PASCAL
SettingsProc(
    HWND hDlg,                              // window handle of the dialog box
    unsigned message,                       // type of message
    WORD wParam,                            // message-specific information
    LONG lParam
    )
{
    char server[STRINGSIZE];

    switch (message) {
        case WM_INITDIALOG:             // message: initialize dialog box
            if(GetPrivateProfileString("HomeDrive", // Section of winfile.ini
                       "Server",                    // Variable
                       "",                          // Default
                       server,                      // Where to put it
                       STRINGSIZE,
                       "winfile.ini"))
                {
                SetDlgItemText(hDlg, IDSERVER, server);
                }
            return (TRUE);

        case WM_COMMAND:                // message: received a command
            switch (wParam)
                {
                case IDOK:
                    GetDlgItemText(hDlg,IDSERVER,server,STRINGSIZE);
                    WritePrivateProfileString("HomeDrive","Server",server,
                                                "winfile.ini");
                    if (ValidBinding)
                        {
                        DestroyBinding();
                        }
                    CreateBinding(server);
                    ValidBinding=TRUE;

                    EndDialog(hDlg, TRUE);          // Exits the dialog box
                    return (TRUE);

                case IDCANCEL:
                    EndDialog(hDlg, FALSE);         // Exits the dialog box
                    return (TRUE);

                default:
                    DefWindowProc(hDlg, message, wParam, lParam);
                }
            break;
    }
    return (FALSE);                                 // Didn't process a message
}

void
CheckRpcStatus(
    char *ProcName,
    MY_STATUS status
    )
{
    char buffer[10];
    char temp[100] = "RPC Exception in ";

    switch (status)
        {
        case RPC_S_INVALID_STRING_BINDING:
        case RPC_S_SERVER_UNAVAILABLE:
        case RPC_S_CALL_FAILED_DNE:
        case RPC_S_ACCESS_DENIED:
            MessageBox(HWND_DESKTOP,
"Check that the HomeDrive server is running on an accessible server.",
                strcat(
                       temp,
                       ProcName
                       ),
                MB_OK);

        case RPC_S_OK:
            break;

        default:
            MessageBox(HWND_DESKTOP,
                _itoa(status,buffer,10),
                strcat(
                       temp,
                       ProcName
                       ),
                MB_OK);
        }
}

void
CheckMyStatus(
    char *ProcName,
    MY_STATUS status
    )
{
    char buffer[10];

    switch (status) {
        case HOMEDIR_S_OK:
            break;

        case HOMEDIR_S_ENTRY_ALREADY_EXISTS:
            MessageBox(HWND_DESKTOP,
                   "That name is already in the database.",
                   "Error",
                   MB_OK);
            break;

        case HOMEDIR_S_ENTRY_NOT_FOUND:
            MessageBox(HWND_DESKTOP,
                   "That name is not in the database.",
                   "Error",
                   MB_OK);
            break;

        default:    // This should never happen
            MessageBox(HWND_DESKTOP,
                   _itoa(status,buffer,10),
                   "Fatal Error",
                   MB_OK);
        }
}

// ====================================================================
//                MIDL allocate and free
// ====================================================================


void *
MIDL_user_allocate(
    size_t len
    )
{
    return(malloc(len));
}

void
MIDL_user_free(
    void * ptr
    )
{
    free(ptr);
}

//***************************************************************************
//
//  AboutDlgProc()
//
//  Purpose:
//
//      Procedure to handle About dialog messages.  This dialog displays
//      copyright and help information.
//
//
//  Parameters:
//
//      IN:
//          hDlg        - Dialog window handle
//          uMessage    - Dialog message
//          wParam      - Message information
//          lParam      - Additional message information
//
//      OUT:
//          N/A
//
//  Return Value:
//
//      Appropriate value for the dialog message.
//
//***************************************************************************

BOOL FAR PASCAL
AboutDlgProc(
    HWND hDlg,
    unsigned uMessage,
    WORD wParam,
    LONG lParam
    )
{
    #define ICONLEFTCOORD 8
    #define ICONTOPCOORD 8
    #define ICONRIGHTCOORD 40
    #define ICONBOTTOMCOORD 40

    #define ICONXLOC 9
    #define ICONYLOC 8

    HDC hdc;
    int i;
    DWORD temp;
    int x;
    int y;
    static HWND    hIcon[16];
    volatile int j;
    static BOOL FirstTime=TRUE;
    static BOOL CookieOn=FALSE;

    switch (uMessage)
    {
        case WM_INITDIALOG:
            for(i=0;i<16;i++)
                hIcon[i]=LoadIcon(hInst,MAKEINTRESOURCE(ANIM1+i));
            return (TRUE);

        case WM_COMMAND:
            switch (wParam)
            {
                case IDOK:
                case IDCANCEL:
                    for(i=0;i<16;i++)
                        DestroyIcon(hIcon[i]);
                    EndDialog (hDlg, TRUE);
                    return (TRUE);

                default:
                    break;
            } // switch (wParam)

        case WM_LBUTTONUP:
        case WM_RBUTTONUP:
            if((CookieOn != (uMessage==WM_LBUTTONUP)) && (wParam & MK_CONTROL))
                {
                hdc=GetDC(hDlg);
                temp=GetCurrentPosition(hdc);
                x=LOWORD(lParam)-LOWORD(temp);
                y=HIWORD(lParam)-HIWORD(temp);
                if((x>ICONLEFTCOORD)&&(x<ICONRIGHTCOORD)&&
                   (y>ICONTOPCOORD)&&(y<ICONBOTTOMCOORD))
                    {
                    if(uMessage==WM_LBUTTONUP)
                        {
                        i=1;
                        }
                    else
                        {
                        i=15;
                        }
                    for(;(i<16) && (i>0);)
                        {
                        DrawIcon(hdc,ICONXLOC,ICONYLOC,hIcon[i]);
                        if(!FirstTime)
                            {
                            for(j=0;j<30000;j++);
                            }
                        if(uMessage==WM_LBUTTONUP)
                            {
                            i++;
                            }
                        else
                            {
                            i--;
                            }
                        }
                    DrawIcon(hdc,ICONXLOC,ICONYLOC,hIcon[0]);
                    FirstTime=FALSE;
                    if(uMessage==WM_LBUTTONUP)
                        {
                        ShowWindow(GetDlgItem(hDlg,ID_OPAQUERECT),SW_SHOWNORMAL);
                        ShowWindow(GetDlgItem(hDlg,ID_CREDITS),SW_SHOWNORMAL);
                        CookieOn=TRUE;
                        }
                    else
                        {
                        ShowWindow(GetDlgItem(hDlg,ID_OPAQUERECT),SW_HIDE);
                        ShowWindow(GetDlgItem(hDlg,ID_CREDITS),SW_HIDE);
                        CookieOn=FALSE;
                        }
                    ReleaseDC(hDlg,hdc);
                    return(TRUE);
                    }
                else
                    {
                    ReleaseDC(hDlg,hdc);
                    }
                }
            break;

        case WM_DESTROY:
            CookieOn=FALSE;

    } // switch (message)

    return (FALSE);
} // AboutDlgProc

void
CreateBinding(
    char *pszNetworkAddress
    )
{
    char * pszUuid             = "12345678-1234-1234-1234-123456789ABC";
    char * pszProtocolSequence = "ncacn_np";
    char * pszEndpoint         = "\\pipe\\home";
    char * pszOptions          = NULL;
    char * pszStringBinding    = NULL;

// FUTURE ENHANCEMENT: use the locator to find the server.
/*
    RPC_NS_HANDLE ImportContext;

    CheckRpcStatus("RpcNsBindingImportBegin",
                    RpcNsBindingImportBegin(
                        RPC_C_NS_SYNTAX_DEFAULT,
                        "/.:/Home",
                        home_ClientIfHandle,
                        NULL,
                        &ImportContext
                        )
                  );

    CheckRpcStatus("RpcNsBindingImportNext",
                    RpcNsBindingImportNext(
                        ImportContext,
                        &home_BindingHandle
                        )
                  );

    CheckRpcStatus("RpcNsBindingImportDone",
                    RpcNsBindingImportDone(&ImportContext)
                  );
*/

//
    CheckRpcStatus("RpcStringBindingCompose",
                    RpcStringBindingCompose(
                        (unsigned char *) pszUuid,
                        (unsigned char *) pszProtocolSequence,
                        (unsigned char *) pszNetworkAddress,
                        (unsigned char *) pszEndpoint,
                        (unsigned char *) pszOptions,
                        (unsigned char * *) &pszStringBinding
                        )
                  );


// Set the binding handle that will be used to bind to the server.

    CheckRpcStatus("RpcBindingFromStringBinding",
                    RpcBindingFromStringBinding(
                        (unsigned char *) pszStringBinding,
                        &home_BindingHandle
                        )
                  );

    CheckRpcStatus("RpcStringFree",
                    RpcStringFree((unsigned char * *) &pszStringBinding)
                  );
//
}

void
DestroyBinding(
    void
    )
{
    CheckRpcStatus("RpcBindingFree",
                RpcBindingFree(&home_BindingHandle));
}

