/*++

Copyright (c) 1989  Microsoft Corporation

Module Name:

    dllerror.c

Abstract:

    This module implements the Error OS/2 V2.0 API Calls

Author:

    Steve Wood (stevewo) 20-Sep-1989 (Adapted from URTL\alloc.c)

Revision History:

--*/

#define INCL_OS2V20_TASKING
#define INCL_OS2V20_ERRORMSG
#define INCL_OS2V20_ERRORS
#include "os2dll.h"
#include "conrqust.h"
#include "os2win.h"


APIRET
DosError(
    IN ULONG ErrorFlags
    )
{
    OS2_API_MSG m;
    POS2_DOSERROR_MSG a = &m.u.DosError;
    ULONG ErrorAction;
    ULONG OriginalErrorAction;

    //
    // Validate the parameter
    //

    if (ErrorFlags & ~(DE_ENABLE_HARD_ERRORS | DE_DISABLE_EXCEPTIONS)) {
        return( ERROR_INVALID_PARAMETER );
        }

    OriginalErrorAction = Od2Process->ErrorAction;

    //
    // Calculate the new ErrorAction bits
    //
    ErrorAction = 0;
    if (ErrorFlags & DE_ENABLE_HARD_ERRORS) {
        ErrorAction |= OD2_ENABLE_HARD_ERROR_POPUP;
        }

    if (!(ErrorFlags & DE_DISABLE_EXCEPTIONS)) {
        ErrorAction |= OD2_ENABLE_ACCESS_VIO_POPUP;
        }


    //
    // Store the new ErrorAction bits
    //

    Od2Process->ErrorAction &= ~(OD2_ENABLE_HARD_ERROR_POPUP |
                                 OD2_ENABLE_ACCESS_VIO_POPUP);
    Od2Process->ErrorAction |= ErrorAction;


    if (Od2Process->ErrorAction != OriginalErrorAction) {
        a->ErrorAction = Od2Process->ErrorAction;

        Od2CallSubsystem( &m, NULL, Os2Error, sizeof( *a ) );
    }

    //
    // Return success
    //

    return( NO_ERROR );
}

typedef struct _ERRORTABLE {
    ULONG ErrorCode : 16;
    ULONG ErrorClass : 8;
    ULONG ErrorAction : 4;
    ULONG ErrorLocus : 4;
} ERRORTABLE, *PERRORTABLE;

ERRORTABLE Od2ErrorTable[] = {
    ERROR_ACCESS_DENIED,            ERRCLASS_AUTH,   ERRACT_USER,   ERRLOC_UNK,
//  ERROR_ADAP_HDW_ERR,             ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_ALREADY_ASSIGNED,         ERRCLASS_ALREADY,ERRACT_USER,   ERRLOC_NET,
    ERROR_ALREADY_EXISTS,           ERRCLASS_ALREADY,ERRACT_USER,   ERRLOC_UNK,
    ERROR_ARENA_TRASHED,            ERRCLASS_APPERR, ERRACT_PANIC,  ERRLOC_MEM,
    ERROR_AUTODATASEG_EXCEEDS_64K,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_BAD_ARGUMENTS,            ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_BAD_COMMAND,              ERRCLASS_INTRN,  ERRACT_PANIC,  ERRLOC_UNK,
//  ERROR_BAD_DEV_TYPE,             ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_NET,
    ERROR_BAD_DRIVER_LEVEL,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_DISK,
    ERROR_BAD_ENVIRONMENT,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_MEM,
    ERROR_BAD_FORMAT,               ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_BAD_LENGTH,               ERRCLASS_INTRN,  ERRACT_PANIC,  ERRLOC_UNK,
//  ERROR_BAD_NETPATH,              ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_NET,
//  ERROR_BAD_NET_NAME,             ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_NET,
//  ERROR_BAD_NET_RESP,             ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_NET,
    ERROR_BAD_PATHNAME,             ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_UNK,
    ERROR_BAD_PIPE,                 ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_UNK,
//  ERROR_BAD_REM_ADAP,             ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_BAD_THREADID_ADDR,        ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_BAD_UNIT,                 ERRCLASS_INTRN,  ERRACT_PANIC,  ERRLOC_UNK,
    ERROR_BAD_EXE_FORMAT,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_BROKEN_PIPE,              ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_BUFFER_OVERFLOW,          ERRCLASS_APPERR, ERRACT_RETRY,  ERRLOC_MEM,
    ERROR_BUSY_DRIVE,               ERRCLASS_ALREADY,ERRACT_DLYRET, ERRLOC_DISK,
    ERROR_CANNOT_MAKE,              ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_DISK,
    ERROR_CALL_NOT_IMPLEMENTED,     ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_CHILD_NOT_COMPLETE,       ERRCLASS_NOTFND, ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_CRC,                      ERRCLASS_MEDIA,  ERRACT_ABORT,  ERRLOC_DISK,
    ERROR_CURRENT_DIRECTORY,        ERRCLASS_AUTH,   ERRACT_USER,   ERRLOC_DISK,
    ERROR_CRITSEC_OVERFLOW,         ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_CRITSEC_UNDERFLOW,        ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_DEV_NOT_EXIST,            ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_DIRECT_ACCESS_HANDLE,     ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_DISK,
    ERROR_DIR_NOT_EMPTY,            ERRCLASS_CANT,   ERRACT_USER,   ERRLOC_DISK,
    ERROR_DIR_NOT_ROOT,             ERRCLASS_CANT,   ERRACT_USER,   ERRLOC_DISK,
    ERROR_DISCARDED,                ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_MEM,
    ERROR_DISK_CHANGE,              ERRCLASS_MEDIA,  ERRACT_USER,   ERRLOC_DISK,
    ERROR_DISK_FULL,                ERRCLASS_OUTRES, ERRACT_USER,   ERRLOC_DISK,
    ERROR_DRIVE_LOCKED,             ERRCLASS_LOCKED, ERRACT_DLYRET, ERRLOC_DISK,
    ERROR_DUP_FCB,                  ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_DUP_NAME,                 ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_DYNLINK_FROM_INVALID_RING,ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TSTDUP,                   ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TSTOVFL,                  ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_EXCL_SEM_ALREADY_OWNED,   ERRCLASS_ALREADY,ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_ENVVAR_NOT_FOUND,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_FAIL_I24,                 ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_FCB_UNAVAILABLE,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_FILE_EXISTS,              ERRCLASS_ALREADY,ERRACT_USER,   ERRLOC_DISK,
    ERROR_FILE_NOT_FOUND,           ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_DISK,
    ERROR_GEN_FAILURE,              ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INFLOOP_IN_RELOC_CHAIN,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INSUFFICIENT_BUFFER,      ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INTERRUPT,                ERRCLASS_UNK,    ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_INVALID_ACCESS,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_AT_INTERRUPT_TIME,ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_BLOCK,            ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_MEM,
    ERROR_INVALID_CALLGATE,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_CATEGORY,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_DATA,             ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_INVALID_DLL_INIT_RING,    ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_DRIVE,            ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_DISK,
    ERROR_INVALID_EVENT_COUNT,      ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_INVALID_EXITROUTINE_RING, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_FLAG_NUMBER,      ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_FUNCTION,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_HANDLE,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_LEVEL,            ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_LIST_FORMAT,      ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_NAME,             ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_UNK,
    ERROR_INVALID_ORDINAL,          ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_PARAMETER,        ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_UNK,
    ERROR_INVALID_PASSWORD,         ERRCLASS_AUTH,   ERRACT_USER,   ERRLOC_UNK,
//  ERROR_INVALID_STARTING_RING,    ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_SEGMENT_NUMBER,   ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_TARGET_HANDLE,    ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_VERIFY_SWITCH,    ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_PROCID,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_PDELTA,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_PCLASS,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_SCOPE,            ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_THREADID,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_STARTING_CODESEG, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_STACKSEG,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_SEGDPL,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_MODULETYPE,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_EXE_SIGNATURE,    ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_MINALLOCSIZE,     ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_FAIL_I24,                 ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_ITERATED_DATA_EXCEEDS_64K,ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_IOPL_NOT_ENABLED,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_LABEL_TOO_LONG,           ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_DISK,
    ERROR_LOCK_VIOLATION,           ERRCLASS_LOCKED, ERRACT_DLYRET, ERRLOC_DISK,
    ERROR_MAX_THRDS_REACHED,        ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOD_NOT_FOUND,            ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_UNK,
    ERROR_MONITORS_NOT_SUPPORTED,   ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MORE_DATA,                ERRCLASS_UNK,    ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_NEGATIVE_SEEK,            ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_DISK,
//  ERROR_NETNAME_DELETED,          ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_NET_WRITE_FAULT,          ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_NET,
//  ERROR_NETWORK_ACCESS_DENIED,    ERRCLASS_AUTH,   ERRACT_USER,   ERRLOC_NET,
//  ERROR_NETWORK_BUSY,             ERRCLASS_TEMPSIT,ERRACT_DLYRET, ERRLOC_NET,
    ERROR_NOT_CURRENT_CTRY,         ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NOT_DOS_DISK,             ERRCLASS_MEDIA,  ERRACT_INTRET, ERRLOC_DISK,
    ERROR_NOT_ENOUGH_MEMORY,        ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_MEM,
    ERROR_NOT_LOCKED,               ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NOT_READY,                ERRCLASS_HRDFAIL,ERRACT_INTRET, ERRLOC_UNK,
    ERROR_NOT_SAME_DEVICE,          ERRCLASS_UNK,    ERRACT_USER,   ERRLOC_DISK,
    ERROR_NOT_SUPPORTED,            ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_NET,
    ERROR_NO_CHILD_PROCESS,         ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NO_DATA,                  ERRCLASS_CANT,   ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_NO_ITEMS,                 ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NO_MORE_FILES,            ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_DISK,
    ERROR_NO_MORE_SEARCH_HANDLES,   ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NO_SIGNAL_SENT,           ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_NO_SPOOL_SPACE,           ERRCLASS_OUTRES, ERRACT_DLYRET, ERRLOC_NET,
    ERROR_NO_PROC_SLOTS,            ERRCLASS_OUTRES, ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_NO_VOLUME_LABEL,          ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_OPEN_FAILED,              ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_UNK,
    ERROR_OUT_OF_PAPER,             ERRCLASS_TEMPSIT,ERRACT_INTRET, ERRLOC_SERDEV,
    ERROR_OUT_OF_STRUCTURES,        ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_PATH_BUSY,                ERRCLASS_ALREADY,ERRACT_USER,   ERRLOC_UNK,
    ERROR_PATH_NOT_FOUND,           ERRCLASS_NOTFND, ERRACT_USER,   ERRLOC_DISK,
    ERROR_PIPE_BUSY,                ERRCLASS_ALREADY,ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_PIPE_NOT_CONNECTED,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_PRINTQ_FULL,              ERRCLASS_OUTRES, ERRACT_DLYRET, ERRLOC_NET,
//  ERROR_PRINT_CANCELLED,          ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_PROC_NOT_FOUND,           ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_PROTECTION_VIOLATION,     ERRCLASS_AUTH,   ERRACT_PANIC,  ERRLOC_UNK,
    ERROR_READ_FAULT,               ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_REDIR_PAUSED,             ERRCLASS_TEMPSIT,ERRACT_RETRY,  ERRLOC_NET,
    ERROR_RELOC_CHAIN_XEEDS_SEGLIM, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_REM_NOT_LIST,             ERRCLASS_TEMPSIT,ERRACT_DLYRET, ERRLOC_NET,
//  ERROR_REQ_NOT_ACCEP,            ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_NET,
    ERROR_RING2SEG_MUST_BE_MOVABLE, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SAME_DRIVE,               ERRCLASS_CANT,   ERRACT_USER,   ERRLOC_UNK,
    ERROR_SECTOR_NOT_FOUND,         ERRCLASS_MEDIA,  ERRACT_ABORT,  ERRLOC_DISK,
    ERROR_SEEK,                     ERRCLASS_HRDFAIL,ERRACT_RETRY,  ERRLOC_DISK,
    ERROR_SEEK_ON_DEVICE,           ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SEM_IS_SET,               ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SEM_NOT_FOUND,            ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SEM_OWNER_DIED,           ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SEM_TIMEOUT,              ERRCLASS_TIME,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SEM_USER_LIMIT,           ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SHARING_BUFFER_EXCEEDED,  ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_MEM,
//  ERROR_SHARING_PAUSED,           ERRCLASS_TEMPSIT,ERRACT_DLYRET, ERRLOC_NET,
    ERROR_SHARING_VIOLATION,        ERRCLASS_LOCKED, ERRACT_DLYRET, ERRLOC_DISK,
    ERROR_SIGNAL_PENDING,           ERRCLASS_CANT,   ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_SIGNAL_REFUSED,           ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SYSTEM_TRACE,             ERRCLASS_CANT,   ERRACT_INTRET, ERRLOC_UNK,
//  ERROR_TOO_MANY_CMDS,            ERRCLASS_OUTRES, ERRACT_DLYRET, ERRLOC_NET,
    ERROR_TOO_MANY_MUXWAITERS,      ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_TOO_MANY_NAMES,           ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_NET,
    ERROR_TOO_MANY_OPEN_FILES,      ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TOO_MANY_SEMAPHORES,      ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TOO_MANY_SEM_REQUESTS,    ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_TOO_MANY_SESS,            ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_NET,
    ERROR_TOO_MANY_TCBS,            ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
//  ERROR_UNEXP_NET_ERR,            ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_NET,
    ERROR_VC_DISCONNECTED,          ERRCLASS_HRDFAIL,ERRACT_RETRY,  ERRLOC_NET,
    ERROR_VIOKBD_REQUEST,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_WAIT_NO_CHILDREN,         ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_WRITE_FAULT,              ERRCLASS_HRDFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_WRITE_PROTECT,            ERRCLASS_MEDIA,  ERRACT_INTRET, ERRLOC_DISK,
    ERROR_WRONG_DISK,               ERRCLASS_MEDIA,  ERRACT_INTRET, ERRLOC_DISK,
    ERROR_NOT_DESCENDANT,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NOT_SESSION_MANAGER,      ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_EXE_MARKED_INVALID,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_UNCERTAIN_MEDIA,          ERRCLASS_INTRN,  ERRACT_ABORT,  ERRLOC_DISK,
    ERROR_FILENAME_EXCED_RANGE,     ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_DISK,
    ERROR_META_EXPANSION_TOO_LONG,  ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_DISK,
    ERROR_RING2_STACK_IN_USE,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_MEM,
    ERROR_CODE_PAGE_NOT_FOUND,      ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_CPLIST_TOO_SMALL,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_CP_NOT_MOVED,             ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_CP_SWITCH_INCOMPLETE,     ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_INVALID_CODE_PAGE,        ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_CANNOT_CREATE_KCB,    ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_CODEPAGE_LOAD_INCOMPL,ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_DETACHED,             ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_FOCUS_ALREADY_ACTIVE, ERRCLASS_ALREADY,ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_KBD_FOCUS_REQUIRED,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_CODEPAGE,     ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_CODEPAGE_ID,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_HANDLE,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_KEYBOARD_BUSY,        ERRCLASS_TEMPSIT,ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_KBD_NO_CODEPAGE_SUPPORT,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_NO_MORE_HANDLE,       ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_UNABLE_TO_FOCUS,      ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MODE_SWITCH_INIT,         ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_DISPLAY_PARMS,      ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INVALID_IOWAIT,     ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INV_ENTRY_PT,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INV_HANDLE,         ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INV_MASK,           ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INVALID_ASCIIZ,     ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INVALID_MASK,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOU_DETACHED,             ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_UNEXPECTED_SLOT_RETURNED, ERRCLASS_INTRN,  ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_DOSSUB_SHRINK,            ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_DOSSUB_NOMEM,             ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_MEM,
    ERROR_DOSSUB_OVERLAP,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_DOSSUB_BADSIZE,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_DOSSUB_BADFLAG,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_DOSSUB_BADSELECTOR,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MR_MSG_TOO_LONG,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MR_MID_NOT_FOUND,         ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MR_UN_ACC_MSGF,           ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MR_INV_MSGF_FORMAT,       ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MR_INV_IVCOUNT,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MR_UN_PERFORM,            ERRCLASS_CANT,   ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_TS_WAKEUP,                ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TS_SEMHANDLE,             ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TS_NOTIMER,               ERRCLASS_OUTRES, ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_TS_HANDLE,                ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_TS_DATETIME,              ERRCLASS_BADFMT, ERRACT_USER,   ERRLOC_UNK,
    ERROR_SYS_INTERNAL,             ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_CURRENT_NAME,         ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_PROC_NOT_OWNED,       ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_PROC_OWNED,           ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_DUPLICATE,            ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_ELEMENT_NOT_EXIST,    ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_NO_MEMORY,            ERRCLASS_OUTRES, ERRACT_DLYRET, ERRLOC_MEM,
    ERROR_QUE_INVALID_NAME,         ERRCLASS_BADFMT, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_INVALID_PRIORITY,     ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_INVALID_HANDLE,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_LINK_NOT_FOUND,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_MEMORY_ERROR,         ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_PREV_AT_END,          ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_PROC_NO_ACCESS,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_EMPTY,                ERRCLASS_CANT,   ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_QUE_NAME_NOT_EXIST,       ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_NOT_INITIALIZED,      ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_UNABLE_TO_ACCESS,     ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_UNABLE_TO_ADD,        ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_UNABLE_TO_INIT,       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_INVALID_MASK,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_PTR,                  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_APTR,                 ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_RPTR,                 ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_CPTR,                 ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_LPTR,                 ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_MODE,                 ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_WIDTH,                ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_ATTR,                 ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_ROW,                  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_COL,                  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_TOPROW,               ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_BOTROW,               ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_RIGHTCOL,             ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_LEFTCOL,              ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_BAD_CP,               ERRCLASS_NOTFND, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_DETACHED,             ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_FONT,                 ERRCLASS_NOTFND, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_ILLEGAL_DURING_POPUP, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_INTERNAL_RESOURCE,    ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_NA_CP,                ERRCLASS_NOTFND, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_NO_CP,                ERRCLASS_NOTFND, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_SHELL_INIT,           ERRCLASS_TEMPSIT,ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_VIO_USER_FONT,            ERRCLASS_APPERR, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_WAIT_FLAG,            ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_UNLOCK,               ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_TRANSPARENT_POPUP,    ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_BAD_RESERVE,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SCS_CALL,                 ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SCS_VALUE,                ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SGS_NOT_SESSION_MGR,      ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_SET_TITLE,            ERRCLASS_CANT,   ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_SMG_BAD_RESERVE,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_BOND_OPTION,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_DATA_LENGTH,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_RELATED_OPT,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_SELECT_OPT,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_START_MODE,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_STOP_OPTION,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_TRACE_OPTION, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_NOT_BOUND,            ERRCLASS_NOTFND, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_SMG_NO_HARD_ERRORS,       ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_PROCESS_NOT_PARENT,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_RETRY_SUB_ALLOC,      ERRCLASS_OUTRES, ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_SMG_SESSION_NON_SELECT,   ERRCLASS_CANT,   ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_SMG_SESSION_NOT_FOREGRND, ERRCLASS_CANT,   ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_SMG_SESSION_NOT_PARENT,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_START_IN_BACKGROUND,  ERRCLASS_ALREADY,ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_KBD_PARAMETER,            ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_NO_DEVICE,            ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_IOWAIT,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_LENGTH,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_ECHO_MASK,    ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_INPUT_MASK,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MON_INVALID_PARMS,        ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MON_INVALID_DEVNAME,      ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MON_INVALID_HANDLE,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MON_BUFFER_TOO_SMALL,     ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MON_BUFFER_EMPTY,         ERRCLASS_NOTFND, ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_MON_DATA_TOO_LARGE,       ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_NO_DEVICE,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INV_PARMS,          ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_CANT_RESET,         ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_INV_MODULE,         ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_NO_DATA,            ERRCLASS_NOTFND, ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_MOUSE_PTR_DRAWN,          ERRCLASS_UNK,    ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_INVALID_FREQUENCY,        ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NLS_NO_COUNTRY_FILE,      ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NLS_OPEN_FAILED,          ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NLS_TABLE_TRUNCATED,      ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NLS_BAD_TYPE,             ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NLS_TYPE_NOT_FOUND,       ERRCLASS_SYSFAIL,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_SMG_ONLY,             ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_INVALID_ASCIIZ,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_DEREGISTER,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_NO_POPUP,             ERRCLASS_APPERR, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_EXISTING_POPUP,       ERRCLASS_LOCKED, ERRACT_DLYRET, ERRLOC_UNK,
    ERROR_KBD_SMG_ONLY,             ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_ASCIIZ,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_INVALID_MASK,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_REGISTER,             ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_KBD_DEREGISTER,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_SMG_ONLY,           ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_REGISTER,           ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_MOUSE_DEREGISTER,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_BAD_ACTION,           ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_CALL,         ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SCS_SG_NOTFOUND,          ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SCS_NOT_SHELL,            ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_INVALID_PARMS,        ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_FUNCTION_OWNED,       ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_RETURN,               ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SCS_INVALID_FUNCTION,     ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SCS_NOT_SESSION_MGR,      ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_REGISTER,             ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_NO_MODE_THREAD,       ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_NO_SAVE_RESTORE_THD,  ERRCLASS_APPERR, ERRACT_IGNORE, ERRLOC_UNK,
    ERROR_VIO_IN_BG,                ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_NOT_BASESHELL,        ERRCLASS_AUTH,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_BAD_STATUSREQ,        ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_PROGRAM_TYPE, ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_PGM_CONTROL,  ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_QUE_INVALID_WAIT,         ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_LOCK,                 ERRCLASS_ALREADY,ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_INVALID_HANDLE,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_ILLEGAL_DURING_LOCK,  ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_VIO_INVALID_LENGTH,       ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_HANDLE_EOF,               ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_HANDLE_DISK_FULL,         ERRCLASS_OUTRES, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_INVALID_SESSION_ID,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_SESSION_NOT_FOUND,    ERRCLASS_NOTFND, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_SMG_NO_SESSIONS,          ERRCLASS_OUTRES, ERRACT_RETRY,  ERRLOC_UNK,
    ERROR_NO_COUNTRY_OR_CODEPAGE,   ERRCLASS_APPERR, ERRACT_ABORT,  ERRLOC_UNK,
    ERROR_NOT_FROZEN,               ERRCLASS_CANT,   ERRACT_ABORT,  ERRLOC_UNK,
    NO_ERROR,                       ERRCLASS_UNK,    ERRACT_ABORT,  ERRLOC_UNK
};

APIRET
DosErrClass(
    IN ULONG ErrorCode,
    OUT PULONG ErrorClass,
    OUT PULONG ErrorAction,
    OUT PULONG ErrorLocus
    )
{
    PERRORTABLE ErrorTableEntry;

    //
    // Scan table of known error codes.  End of table is marked with NO_ERROR
    // entry.
    //

    ErrorTableEntry = &Od2ErrorTable[ 0 ];
    while (ErrorTableEntry->ErrorCode != NO_ERROR) {
        if (ErrorTableEntry->ErrorCode == ErrorCode) {
            break;
            }
        else {
            ErrorTableEntry++;
            }
        }


    //
    // Either entry found or default NO_ERROR entry found.  Fill in the three
    // out parameters with the Error Class, Action and Locus information from
    // the table entry.
    //

    try {
        *ErrorClass = ErrorTableEntry->ErrorClass;
        *ErrorAction = ErrorTableEntry->ErrorAction;
        *ErrorLocus = ErrorTableEntry->ErrorLocus;
    } except( EXCEPTION_EXECUTE_HANDLER ) {
       Od2ExitGP();
    }

    //
    // Return success
    //

    return( NO_ERROR );
}

typedef struct _X_FILE_NAME_INFORMATION {
    FILE_NAME_INFORMATION f;
    char x[255];
} X_FILE_NAME_INFORMATION;

typedef struct _X_OBJECT_NAME_INFORMATION {
    OBJECT_NAME_INFORMATION f;
    char x[255];
} X_OBJECT_NAME_INFORMATION;

BOOLEAN
RetryIO(
    NTSTATUS Status,
    HANDLE Handle
    )
{
    NTSTATUS       LocalStatus;
//  IO_STATUS_BLOCK IoStatus;
//  X_FILE_NAME_INFORMATION FileNameInfo;
    X_OBJECT_NAME_INFORMATION ObjectNameInfo;
    ULONG   ReturnedSize;
    UNICODE_STRING Name_U;
    int     ReturnedAction, Drive;
    BOOLEAN WriteProtectError;

    if (!((Od2Process->ErrorAction & OD2_ENABLE_HARD_ERROR_POPUP) &&
          (
           (Status == STATUS_NO_MEDIA_IN_DEVICE) ||
           (Status == STATUS_DEVICE_NOT_READY) ||
           (Status == STATUS_MEDIA_WRITE_PROTECTED)
          )
         )
       ) {
        return (FALSE);
    }

    LocalStatus = NtQueryObject(Handle,
                                ObjectNameInformation,
                                &ObjectNameInfo,
                                sizeof(ObjectNameInfo),
                                &ReturnedSize
                               );
    if (!NT_SUCCESS(LocalStatus)) {
#if DBG
        DbgPrint("HardError - Status = %lx\n", LocalStatus);
#endif
        Drive = 0; // No Drive information
    }
    else {
#if DBG
        DbgPrint("HardError - ObjectName = %ws\n", ObjectNameInfo.f.Name.Buffer);
#endif
        ObjectNameInfo.f.Name.Length = 16; // "\Device\Floppy0\"
        RtlInitUnicodeString(&Name_U, L"\\Device\\Floppy0\\");
        Name_U.Length = 16;
        if (RtlEqualUnicodeString(&ObjectNameInfo.f.Name, &Name_U, TRUE)) {
            Drive = 1; // This is Drive A:
#if DBG
            DbgPrint("HardError - Accessing drive A:\n");
#endif
        }
        else {
            RtlInitUnicodeString(&Name_U, L"\\Device\\Floppy1\\");
            Name_U.Length = 16;
            if (RtlEqualUnicodeString(&ObjectNameInfo.f.Name, &Name_U, TRUE)) {
                Drive = 2; // This is Drive B:
#if DBG
                DbgPrint("HardError - Accessing drive B:\n");
#endif
            }
            else {
                Drive = 0; // No Drive information
            }
        }
    }

    WriteProtectError = (Status == STATUS_MEDIA_WRITE_PROTECTED) ?
                                        (BOOLEAN)TRUE : (BOOLEAN)FALSE;

    LocalStatus = Ow2HardErrorPopup(
                                   Drive,
                                   WriteProtectError,
                                   &ReturnedAction,
                                   &Od2Process->ApplName[0]
                                  );

    if (!NT_SUCCESS(LocalStatus)) {
        ASSERT(NT_SUCCESS(LocalStatus));
        return (FALSE);
    }

    if (ReturnedAction == OS2SS_IDIGNORE) {
        return (FALSE);
    }
    else if (ReturnedAction == OS2SS_IDABORT) {
        Od2DosExit( EXIT_PROCESS, 0, TC_HARDERROR );
    }
    else { // OS2SS_RETRY
        return (TRUE);
    }
}

BOOLEAN
RetryCreateOpen(
    NTSTATUS Status,
    POBJECT_ATTRIBUTES pObja
    )
{
    NTSTATUS       LocalStatus;
    IO_STATUS_BLOCK IoStatus;
    X_FILE_NAME_INFORMATION FileNameInfo;
    int     ReturnedAction, Drive;
    BOOLEAN WriteProtectError;

    if (!((Od2Process->ErrorAction & OD2_ENABLE_HARD_ERROR_POPUP) &&
          (
           (Status == STATUS_NO_MEDIA_IN_DEVICE) ||
           (Status == STATUS_DEVICE_NOT_READY) ||
           (Status == STATUS_MEDIA_WRITE_PROTECTED)
          )
         )
       ) {
        return (FALSE);
    }

    if (pObja->RootDirectory != NULL) {
        LocalStatus = NtQueryInformationFile(pObja->RootDirectory,
                                        &IoStatus,
                                        &FileNameInfo,
                                        sizeof(X_FILE_NAME_INFORMATION),
                                        FileNameInformation
                                       );
        if (!NT_SUCCESS(LocalStatus)) {
#if DBG
            DbgPrint("HardError - Status = %lx\n", LocalStatus);
#endif
            Drive = 0; // No Drive information
        }
        else {
#if DBG
            DbgPrint("HardError - Filename = %ws\n", FileNameInfo.f.FileName);
#endif
            Drive = 0; // No Drive information
        }
    }
    else {
        Drive =
            RtlUpperChar((CHAR)pObja->ObjectName->Buffer[FILE_PREFIX_LENGTH+DRIVE_LETTER]) - 'A' + 1;
    }

    WriteProtectError = (Status == STATUS_MEDIA_WRITE_PROTECTED) ?
                                        (BOOLEAN)TRUE : (BOOLEAN)FALSE;

    LocalStatus = Ow2HardErrorPopup(
                                   Drive,
                                   WriteProtectError,
                                   &ReturnedAction,
                                   &Od2Process->ApplName[0]
                                  );

    if (!NT_SUCCESS(LocalStatus)) {
        ASSERT(NT_SUCCESS(LocalStatus));
        return (FALSE);
    }

    if (ReturnedAction == OS2SS_IDIGNORE) {
        return (FALSE);
    } else if (ReturnedAction == OS2SS_IDABORT) {
        Od2DosExit( EXIT_PROCESS, 0, TC_HARDERROR );
    }
    else {
        return (TRUE);
    }
}

