
//--------------------------------------------------------------------------
//
// Module Name:  FILTER.C
//
// Brief Description:  This module contains the PSCRIPT driver's routines
// for implementing Level2 Filters RLE and ASCII85. It also contains the
// Hex encoding filter used by both the Level 1 and Level 2 code
//
// Author:  James Bratsanos (v-jimbr)
// Created: 28-Apr-1994
//
// Copyright (c) 1990-1994 Microsoft Corporation
//
//--------------------------------------------------------------------------

#include "pscript.h"
#include "filter.h"


BYTE  FILTER_CRLF[] = { 0x0a };


//
// The precomputes values used in converting base 256 to base 85
//

DWORD ConstAscii85[] = {
    52200625L,
      614125L,
        7225L,
          85L
};


//
// Source starts here
//

WORD
FilterConvertToASCII85(
    PBYTE pIn,
    PBYTE pbConvert,
    BOOL  bZeroOpt
)

/*++

Routine Description:

    This function takes a buffer with 4 bytes of binary data and converts it to
    the ascii85 postscript level ii format.

    Each set of 4 bytes is converted to 5 bytes. This results in a
    20% increase in stream size, but still keeps everything in the printable
    ascii range. The conversion is done from base 256 to base 85 as follows

    byte1 * 256^3 + byte2 * 256^2 + byte3 * 256 + byte 4 =

    NewByte1 * 85^4 + NewByte2 * 85^3 + NewByte3 * 85^2 + NewByte4 * 85 + NewByte5

    Each NewByte1 is then added to ascii 33 to create a printable character.
    As a special case if all NewBytes* are 0 they are replaced by the single
    character "z".


Arguments:

    pIn         - Pointer to a buffer containing 4 bytes. ONLY the first 4 bytes
                  are converted.

    pbConvert   - Pointer to destination buffer, which will be filled with
                  either 5 or 1 bytes.

    bZeroOpt    - TRUE if the function is allowed to use 0's optimization which
                  generates a single "z" if all generated bytes are 0.


Return Value:

    The number of bytes generated into pbConvert.

Author:

    29-Apr-1994 created  -by-  James Bratsanos (v-jimbr)


Revision History:


--*/

{
    DWORD dwNewNum, dwTmp;
    PDWORD pConst;
    BYTE   i;



    //
    // Generate the number to work with by multiplying by 256^<3,2,1>
    //

    dwNewNum = *pIn++ * 16777216L;

    dwNewNum += *pIn++ * 65536L;

    dwNewNum += *pIn++ * 256L;

    dwNewNum += *pIn++;


    //
    // Now generate the new bytes, taking care of the special case of
    // all 0's
    //

    if (!bZeroOpt || dwNewNum ) {

        for ( i = 0, pConst = ConstAscii85;
                i < 4  ;
            i++, pConst++ ) {
            dwTmp    = dwNewNum / *pConst;
            dwNewNum -= (*pConst * dwTmp);
            *pbConvert++ = (BYTE) dwTmp + '!';

        }

        *pbConvert = (BYTE) dwNewNum + '!';
        return 5;

    } else {

        //
        // All zeros, and zero optimization is turned on so use it
        //

        *pbConvert   = 'z';
        return 1;
    }

}




DWORD
FilterGenerateFlags(
    PDEVDATA pdev
)

/*++

Routine Description:

    This function generates the filter flags to use based on whether
    the user has requested bitmap compression (AdvancedDocumentProperties)
    and whether or not the printer is capable of it

Arguments:

    pdev        - Pointer to the PDEVDATA data structure to determine what
                  type of postscript output for current device

Return Value:

    The flags to pass into the FilterInit function.

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)


Revision History:


--*/
{
    DWORD dwRet;

    //
    // Here we determine what type of filtering to use based on the language
    // level of the printer
    // NOTE:!!!!! you can ONLY use FILTER_FLAG_HEX by itself... it may
    //            NOT be combined with anything ELSE!!!! this is because the
    //            headers created by adobe always assume that leve on ascii
    //            is never done via a filter!!!
    //

    if (Level2Device(pdev->hppd) &&
        (pdev->dm.dmPrivate.dwFlags & PSDEVMODE_COMPRESSBMP))
    {

        //
        // The printer is capable of filter (level 2) and the user has
        // requested bitmaps be compressed.
        //

        dwRet = FILTER_FLAG_RLE | FILTER_FLAG_ASCII85;

    } else {

        dwRet = FILTER_FLAG_HEX;
    }

    return( dwRet );
}

VOID
FilterGenerateFilterProc(
    PFILTER pFilter
)

/*++

Routine Description:

    This function outputs the required postscript code to create a file
    object that the postscript interpreter will read from. The code
    is output directly from here.

Arguments:

    pFilter    - Pointer to a pFilter structure

Return Value:

    VOID

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:


--*/
{

    //
    // If the flag is not hex then were doing level two style filters,
    // by actually using the level ii filter command. HEX is always done
    // in the level one style.
    //

    if (!(pFilter->dwFilterFlags & FILTER_FLAG_HEX )) {


        psputs(pFilter->pdev, "/DrvPIn currentfile ");

        //
        // Note: The hex filter is missing from here, because its not currently
        //       supported because of the way we send data to the printer, if
        //       were letting the printer do the halftoning.
        //

        if (pFilter->dwFilterFlags & FILTER_FLAG_ASCII85 ) {
            psputs( pFilter->pdev, "/ASCII85Decode filter");
        }


        if (pFilter->dwFilterFlags & FILTER_FLAG_RLE ) {
            psputs( pFilter->pdev, "/RunLengthDecode filter");
        }

        psputs( pFilter->pdev, " def ");

    }

}

VOID
FilterGenerateImageProc(
    PFILTER pFilter,
    BOOL    bColor
)

/*++

Routine Description:

    This function generates the postscript code to correctly image a bitmap
    based on the filter flags, the postscript level of the target device and
    the color format of the source bitmap. This is more complicated than it
    needs to be because the adobe header thats used when we let the printer
    do the halftoning always assumes HEX is in the level one mode.

Arguments:

    pFilter     - Pointer to a pFilter structure

    bColor      - True if the source bitmap is color

Return Value:

    VOID

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:


--*/
{
    extern BYTE rgbm[];
    INT         i;
    LPBYTE      prgbm;


    if (bColor) {

        i     = 3;
        prgbm = rgbm;

    } else {

        i     = 1;
        prgbm = &rgbm[3];
    }

    if ( pFilter->dwFilterFlags & FILTER_FLAG_HEX )  {


        //
        // This is always treated as level one mode regardless of the
        // real poscript level.
        //

        while (i--) {

            psprintf(pFilter->pdev,
                     "\n{currentfile %cstr readhexstring pop}", *prgbm++);
        }

        psputs(pFilter->pdev, (bColor) ? "\n" : " ");

    } else {

        if (bColor) {

            //
            // If the flags specify non hex its either level two or Binary
            // (Level one or Level 2) so assume the DrvPIn object is around
            //

            while (i--) {

                psprintf(pFilter->pdev,
                         "\n{DrvPIn %cstr readstring pop}", *prgbm++);
            }

            psputs(pFilter->pdev, "\n");

        } else {

            if (Level2Device(pFilter->pdev->hppd)) {

                //
                // If were level two, then we can read directly from the
                // source which has been defined earlier.
                //

                psputs( pFilter->pdev, "\nDrvPIn ");

            } else {

                //
                // Level one requires we supply a procedure!!! to read from
                // so do it now..
                //

                psputs(pFilter->pdev, "\n{currentfile mstr readstring pop} ");
            }
        }
    }

    //
    // Finally send out the correct proc
    //

    if (bColor) {

        psputs(pFilter->pdev, "true 3 colorimage\n");

    } else if (pFilter->pdev->dwFlags & PDEV_DOIMAGEMASK) {

        psputs(pFilter->pdev, "imagemask\n");

    } else {

        psputs(pFilter->pdev, "image\n");
    }
}


BOOL
FilterWriteBinary(
    PFILTER pFilter,
    PBYTE   pIn,
    LONG    lCount
)

/*++

Routine Description:

    This function simply passes the data on untouched to the normal output
    function. This should always be the LAST filter used if filters are
    chained.

    If we ever implement BCP or TBCP this would be the place to do
    the conversion.

Arguments:

    pFilter     - Pointer to a filter structure.

    pIn         - Pointer to source buffer

    lCount      - Count of bytes in source buffer

Return Value:

    BOOLEAN true if the write was succesful, FALSE if the user aborted.

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:


--*/
{
    BOOL ok=TRUE;

    //
    // This routine knows how to send real data out the printer
    //
    if (pIn) {
        ok =  pswrite(pFilter->pdev, pIn, lCount );
    }

    return(ok);
}

BOOL
FilterWriteRLE(
    PFILTER pFilter,
    PBYTE   pIn,
    LONG    lCount
)

/*++

Routine Description:

    This function converts the input buffer to a RLE (Run Length Encoded)
    format. The encoding is defined as follows.

    The output record consists of a control byte followed by up to 128 bytes
    of data. The control byte is defined as:

    n = 0 -> 127   :: The next n+1 bytes should be copied literally.

    n = 129 -> 255 :: The next byte should be repeated 257 - n times

    n = 128        :: EOD (End of data) this closes the filter.

Arguments:

    pFilter     - Pointer to a filter structure.

    pIn         - Pointer to source buffer

    lCount      - Count of bytes in source buffer

Return Value:

    BOOLEAN true if the write was succesful, FALSE if the user aborted.

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)
    22-May-1994  -by-  James Bratsanos (a-jimbr)
        Fixed the break out logic to work correctly if the job is canceled.

Revision History:


--*/
{
    BYTE    tempByte[2];
    PBYTE   pbSrcBeg;
    PBYTE   pbSrcEnd;
    PBYTE   pbLastRepeat;
    PBYTE   pbTmp;
    BYTE    LastSrc;

    LONG    RepeatCount;
    LONG    LiteralCount;
    LONG    CurSize;
    BOOL    ok=TRUE;




    //
    // First check to see if the src pointer is 0, if it is we need to send
    // the RLE EOD marker then pass on the flush case to the next filter
    //

    if (!pIn) {


        //
        // EOD (end of data) this will cancel the level 2 filter
        //

        tempByte[0] = (BYTE) RLE_EOD;

        ok = (*pFilter->rleInfo.fcOutData) ( pFilter, tempByte, 1);


        //
        // And now pass the end of data on
        //

        ok = (*pFilter->rleInfo.fcOutData) ( pFilter, pIn, lCount);

    } else {

        pbSrcBeg     = pIn;
        pbSrcEnd     = pIn + lCount;

        pbLastRepeat = pbSrcBeg;

        while ( ok && (pbSrcBeg < pbSrcEnd) ) {

            pbTmp   = pbSrcBeg;
            LastSrc = *pbTmp++;

            while ((pbTmp < pbSrcEnd) &&
                   (*pbTmp == LastSrc)) {

                ++pbTmp;
            }

            if (((RepeatCount = (LONG)(pbTmp - pbSrcBeg)) >= RLE_MIN_REPEATS) ||
                (pbTmp >= pbSrcEnd)) {


                LiteralCount = (LONG)(pbSrcBeg - pbLastRepeat);


                if (RepeatCount < RLE_MIN_REPEATS) {

                    //
                    // If repeats is not good enough to make it into repeat control
                    // then add it to the literal count
                    //

                    LiteralCount += RepeatCount;
                    RepeatCount   = 0;
                }


                //
                // Setting literal count
                //

                while (LiteralCount && ok) {

                    if ((CurSize = LiteralCount) > RLE_MAX_LITERAL) {

                        CurSize = RLE_MAX_LITERAL;
                    }

                    //
                    // Set literal control bytes from 0-127
                    //

                    tempByte[0] = (BYTE)(CurSize - 1);

                    (*pFilter->rleInfo.fcOutData) ( pFilter, tempByte, 1);

                    ok = (*pFilter->rleInfo.fcOutData) ( pFilter, pbLastRepeat, CurSize);

                    pbLastRepeat += CurSize;
                    LiteralCount -= CurSize;
                }

                //
                // Setting repeat count if any
                //

                while (RepeatCount && ok) {

                    if ((CurSize = RepeatCount) > RLE_MAX_REPEATS) {

                        CurSize = RLE_MAX_REPEATS;
                    }


                    //
                    // Set Repeat Control bytes from -1 to -127
                    //

                    tempByte[0] = (BYTE)( 1  - CurSize );
                    tempByte[1] = (BYTE) LastSrc;

                    ok = (*pFilter->rleInfo.fcOutData) ( pFilter, tempByte, 2);

                    //
                    // If we have more than RLE_MAX_REPEATS then we want to make
                    // sure we used most efficient method to send.  If we have
                    // remaining repeated bytes less than RLE_MIN_REPEATS then
                    // we want to skip that and use literal for the next run
                    //

                    if ((RepeatCount -= CurSize) < RLE_MIN_REPEATS) {


                        pbTmp       -= RepeatCount;
                        RepeatCount  = 0;
                    }
                }

                pbLastRepeat = pbTmp;
            }

            pbSrcBeg = pbTmp;
        }
    }

    return(ok);
}




BOOL
FilterWriteASCII85(
    PFILTER pFilter,
    PBYTE   pIn,
    LONG    lCount
)

/*++

Routine Description:

    This function encodes the input buffer to the ASCII85 Postscript level 2
    filter by calling the function FilterConvertToASCII85(...). See this function
    for a description of the conversion mechanism

    As a special case when the filter is closed with n bytes left n = 1,3 then
    the conversion is done by filling the original sequence with 4 - n bytes of
    zeros, the doing the conversion normally. BUT only n+1 bytes are output
    followed by the ~> EOD (end of data) marker.

    NOTE: This filter MUST be closed by calling it with a NULL pIn pointer.

    NOTE: This filter also wraps line every FILTER_WRAP_COUNT columns


Arguments:

    pFilter     - Pointer to a filter structure.

    pIn         - Pointer to source buffer

    lCount      - Count of bytes in source buffer

Return Value:

    BOOLEAN true if the write was succesful, FALSE if the user aborted.

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:

--*/
{
    BYTE            bConvert[5];
    PBYTE           pbDst;
    PASCII85INFO    pInfo;
    DWORD           dwAmtToMove;
    BOOL            ok=TRUE;
    WORD            wNumGenerated;



    pInfo = &pFilter->ascii85Info;


    //
    // If the current in storage plus whats passed in is not enough just
    // save it in the buff since we need at least 4 bytes to do any work.
    //

    if (pIn && pInfo->ascii85Cnt + lCount < 4 ) {


        ASSERTMSG(lCount, "WriteASCII85 have buffer but 0 count");

        memcpy(pInfo->ascii85Buff + pInfo->ascii85Cnt,
                   pIn,
                   lCount);

        pInfo->ascii85Cnt += (WORD) lCount;

    } else {

        //
        // First we need to see if we have any data left over
        //

        if (pInfo->ascii85Cnt) {

            //
            // Handle the closing filter case.
            //

            if (!pIn) {
                WORD wAmtToWrite;

                //
                // This is the flushing case so we need to complete the last one
                // by writing zeros into the missing entries
                //

                pbDst = pInfo->ascii85Buff + pInfo->ascii85Cnt;

                dwAmtToMove = 4 - pInfo->ascii85Cnt;

                wAmtToWrite = pInfo->ascii85Cnt + 1;

                while (dwAmtToMove--) {
                    *pbDst++ = 0;
                }

                pInfo->ascii85Cnt = 0;

                //
                // Convert the number, not allowing the all zero optimization
                // from occuring
                //

                wNumGenerated = FilterConvertToASCII85( pInfo->ascii85Buff,
                                                        bConvert,
                                                        FALSE );


                ok = (*pInfo->fcOutData) (pFilter, bConvert, wAmtToWrite);


            } else {

                //
                // Copy enough to do one 5 tupple then go on
                //

                dwAmtToMove = 4 - pInfo->ascii85Cnt;


                ASSERTMSG(dwAmtToMove, "WriteASCII85 have data in buff == 4");

                if (dwAmtToMove) {
                    memcpy(pInfo->ascii85Buff + pInfo->ascii85Cnt,
                               pIn,
                               dwAmtToMove);

                    pIn += dwAmtToMove;
                    lCount -= dwAmtToMove;
                }

                pInfo->ascii85Cnt = 0;


                //
                // Now output just the one..
                //

                ok = FilterWriteASCII85( pFilter, pInfo->ascii85Buff, 4);



            }

        }


        // If this is not the last case then a real buffer has been passed in
        // so act on it
        //

        if (pIn) {

            while (ok &&  lCount > 3) {

                //
                // Convert the 4 bytes until we have less than 4 left.
                //

                wNumGenerated = FilterConvertToASCII85( pIn, bConvert, TRUE );

                //
                // Write out the converted data
                //

                ok = (*pInfo->fcOutData) (pFilter, bConvert, wNumGenerated);

                //
                // Bump the pointes and decrement the count
                //

                pIn+=4;
                lCount -= 4;


                //
                // Check to see if we need to wrap the columns
                //

                pInfo->colCnt += wNumGenerated;

                if (pInfo->colCnt >= FILTER_WRAP_COUNT) {
                    pInfo->colCnt = 0;

                    ok = (*pInfo->fcOutData) (  pFilter,
                                                FILTER_CRLF,
                                                sizeof(FILTER_CRLF) );

                }


            }  // While there are at least 4 bytes of src data to act on
        }

        //
        // Do this follow up code ONLY if everything is okay, if not
        // just break out..
        //

        if (ok) {

            if (!pIn) {

                ASSERTMSG( (pInfo->ascii85Cnt == 0 ) ,
                          "WriteASCII85 close case but buffer not empty");

                //
                // Flushing buffer case, the previous code should have
                // exhausted our buffer, now we output the EOD and pass the
                // close case on.
                //

                (*pInfo->fcOutData) (pFilter, "~>", 2);

                ok = (*pInfo->fcOutData) (pFilter, pIn, lCount);

            } else if (lCount) {

                ASSERTMSG((pInfo->ascii85Cnt == 0 ) ,
                          "WriteASCII85 need to save but buff not empty");

                ASSERTMSG( (lCount < 4 ),
                          "WriteASCII85 final output has more than 4 bytes left?");

                //
                // If there was anything left over we have to save it...
                //

                memcpy(pInfo->ascii85Buff, pIn, lCount);
                pInfo->ascii85Cnt = (WORD) lCount;

            }

        }

    }

    return(ok);
}


BOOL
FilterWriteHex(
    PFILTER pFilter,
    PBYTE   pIn,
    LONG    lCount
)

/*++

Routine Description:

    This filter converts binary to hex by converting each byte to two
    ascii bytes that represent the original byte. ie 0 > "00" and FF becomes
    "FF". This filter doubles the size of the original stream but guarantees
    that all bytes in the range of 0-255 are in the ASCII printable range.

    NOTE:!!!!!!!!! this filter can't be combined with other filters, and
                   whenever this filter is used its assumed to be in the
                   level one mode!!!!!!! dont ever use this in the level
                   two mode (ie a proc that reads from the a filter!!!
                   because it will not work!)


    NOTE: This filter also wraps line every FILTER_WRAP_COUNT columns

Arguments:

    pFilter     - Pointer to a filter structure.

    pIn         - Pointer to source buffer

    lCount      - Count of bytes in source buffer

Return Value:

    BOOLEAN true if the write was succesful, FALSE if the user aborted.

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:

--*/
{
    CHAR buf[2];
    BOOL ok = TRUE;
    int b1, b2;

    //
    // If this is the flush case then handle it
    //

    if (!pIn) {

        //
        // If this was a REAL level two filter, then we have two terminate
        // the filter with the correct character. Then pass on the flushing
        // state to the next filter
        //


        ok = (*pFilter->hexInfo.fcOutData) (pFilter, pIn, lCount);

    } else {

        while (lCount > 0 && ok) {

            // Get a byte and convert it to two hex digits

            b1 = *pIn;
            b2 = (b1 >> 4) & 0x0f;      /* leading digit */
            b1 &= 0x0f;                 /* trailing digit */

    //!!! during the code review it was suggested to use itoa here.
    //!!! is that really worth doing??? - kentse.

            if (b2 >= 10)
                b2 += ('A' - 10);
            else
                b2 += '0';

            if (b1 >= 10)
                b1 += ('A' - 10);
            else
                b1 += '0';

            buf[0] = (CHAR) b2;
            buf[1] = (CHAR) b1;
        
            ok = (*pFilter->hexInfo.fcOutData) (pFilter, buf, 2);

            --lCount;
            ++pIn;


            //
            // See if we need to wrap columns
            //

            pFilter->hexInfo.colCnt += 2;

            if (pFilter->hexInfo.colCnt >= FILTER_WRAP_COUNT) {
                pFilter->hexInfo.colCnt = 0;

                ok = (*pFilter->hexInfo.fcOutData) ( pFilter,
                                                     FILTER_CRLF,
                                                     sizeof(FILTER_CRLF) );

            }


          }

    }

    return( ok );
}

SHORT
FilterPSBitMapType(
    PFILTER pFilter,
    BOOL BinaryClearChannel
)

/*++

Routine Description:

    This function generates a number which designates the procedure to
    read from for the beginimage procedure defined in the postscript image
    header. This routine currently is not complete as there are combinations
    of filters that dont yet exist in the headers ie ASCII85 only...

Arguments:

    pFilter            - Pointer to a filter structure.

    BinaryClearChannel - True if the destination chanel is clear ie any byte
                         0-255 can be sent transparently

Return Value:

    SHORT either 0,1,2,3 where

         0       | Unpacked binary data               - level1 or level2
         1       | Unpacked hex    data               - level1 or level2
         2       | Packed binary data( RLE only)      - level2
         3       | Packed ascii85 data(RLE + ascii85) - level2

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:


--*/

{
    PDEVDATA pDev;

    pDev = pFilter->pdev;


    if(!BinaryClearChannel)
    {

        if ((pFilter->dwFilterFlags & FILTER_FLAG_RLE) &&
            (pFilter->dwFilterFlags & FILTER_FLAG_ASCII85 )) {
            return(3);
        } else {
            return(1);
        }

    } else {

        if (pFilter->dwFilterFlags & FILTER_FLAG_RLE ) {
            return(2);
        } else {
            return(0);
        }

    }
}


VOID
FilterInit(
    PDEVDATA pdev,
    PFILTER  pFilter,
    DWORD    dwFlags
)

/*++

Routine Description:

    This function initializes the passed in FILTER object based on the
    flags passed in. This generates a filter object to which data can
    be written and in this way filters can be chained together to form
    multiple encoding filters. The order however is fixed and should
    not be changed. ie if both RLE and ASCII85 are specified RLE is called
    first followed by ASCII85. Once a filter has been initialized you can
    write to it by calling the FILTER_WRITE macro. After writing is complete
    you MUST close the filter by calling FILTER_WRITE with a NULL input
    buffer and a count of 0. You MUST re-call FilterInit to write via the filter
    mechanism again, after closing.

Arguments:

    pdev        - Pointer to the PDEVDATA data structure to determine what
                  type of postscript output for current device

    pFilter     - engine SURFOBJ to be examine

    dwFlags     - What encoding to enable supported are:

                    FILTER_RLE      - Run length encoding
                    FILTER_ASCII85  - ASCII85 encoding
                    FILTER_HEX      - HEX encoding (ONLY by itself!!!)

Return Value:

    VOID

Author:

    29-Apr-1994  -by-  James Bratsanos (v-jimbr)

Revision History:


--*/
{
    PFILTERFUN *pNextToCall;



    //
    // This init is responsible for setting up the correct filter mechanism
    // based on the passed in flags. We will respect the convention of having
    //

    ASSERTMSG(pdev, "FilterInit: NULL pdev\n");
    ASSERTMSG(pFilter, "FilterInit: NULL pFilter\n");
    ASSERTMSG(dwFlags,"FilterInit: No flags specified\n");

    pNextToCall = &pFilter->fcInitialFilter;

    pFilter->pdev = pdev;
    pFilter->dwFilterFlags = 0;



    if (dwFlags & FILTER_FLAG_RLE) {
        //
        // We have an RLE filter so set up its data..
        //

        //
        // With RLE filters they can only output binary or ASCII85 so check now
        //
        *pNextToCall  = FilterWriteRLE;

        pNextToCall = &pFilter->rleInfo.fcOutData;

        pFilter->dwFilterFlags |= FILTER_FLAG_RLE;
    }

    if (dwFlags & FILTER_FLAG_ASCII85) {

        pFilter->ascii85Info.ascii85Cnt = 0;
        pFilter->ascii85Info.colCnt = 0;


        *pNextToCall = FilterWriteASCII85;

        pNextToCall = &pFilter->ascii85Info.fcOutData;

        pFilter->dwFilterFlags |= FILTER_FLAG_ASCII85;

    }


    if (dwFlags & FILTER_FLAG_HEX) {


        ASSERTMSG( ((dwFlags & FILTER_FLAG_HEX ) == dwFlags),
                  "FilterInit: The HEX filter may NOT be mixed with other filters");

        //
        // Nothing to set up except the call functions, and set up the col count
        // so we know to output a CR every 80 colums
        //

        pFilter->hexInfo.colCnt = 0;

        *pNextToCall = FilterWriteHex;
        pNextToCall = &pFilter->hexInfo.fcOutData;

        pFilter->dwFilterFlags |= FILTER_FLAG_HEX;
    }

    //
    // Last one is ALWAYS binary
    //

    *pNextToCall = FilterWriteBinary;


}

