/*++

Copyright (c) 1992  Microsoft Corporation

Module Name:

    ImpDel.c

Abstract:

    NetrReplImportDirDel is the local worker for the NetReplImportDirDel API.

Author:

    John Rogers (JohnRo) 09-Jan-1992

Environment:

    Runs under Windows NT.
    Requires ANSI C extensions: slash-slash comments, long external names.

Revision History:

    09-Jan-1992 JohnRo
        Created.
    20-Jan-1992 JohnRo
        Netr prototypes are now generated by MIDL and put in repl.h.
    27-Jan-1992 JohnRo
        Changed to use LPTSTR etc.
    15-Mar-1992 JohnRo
        Update registry with new values.
    22-Jul-1992 JohnRo
        RAID 2274: repl svc should impersonate caller.
    12-Nov-1992 JohnRo
        RAID 1537: repl APIs in wrong role kill service.
        Also fix remote repl admin.

--*/


// These must be included first:

#include <windef.h>             // IN, VOID, LPTSTR, etc.
#include <lmcons.h>             // NET_API_STATUS, PARM equates, etc.
#include <repldefs.h>           // (Needed by client.h)
#include <rpc.h>                // Needed by <repl.h>.

// These can be in any order:

#include <client.h>             // ReplRemoveClientRecForDirName().
#include <dirname.h>            // ReplIsDirNameValid().
#include <impdir.h>             // ImportDirDeleteConfigData().
#include <lmerr.h>      // NERR_, ERROR_ equates, NO_ERROR.
#include <netlock.h>            // ACQUIRE_LOCK(), etc.
#include <repl.h>               // My prototype (in MIDL-generated .h file).
#include <replgbl.h>    // ReplConfigLock, ReplConfigRole.
#include <rpcutil.h>    // NetpImpersonateClient(), NetpRevertToSelf().


NET_API_STATUS
NetrReplImportDirDel (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName
    )

/*++

Routine Description:

    Same as NetReplImportDirDel.

Arguments:

    Same as NetReplImportDirDel.

Return Value:

    Same as NetReplImportDirDel.

--*/

{
    NET_API_STATUS ApiStatus;
    BOOL Impersonated = FALSE;
    BOOL LockedClientList = FALSE;
    BOOL LockedConfigData = FALSE;

    if (ReplIsDirNameValid( DirName ) == FALSE) {
        return (ERROR_INVALID_PARAMETER);
    }

    //
    // Impersonate caller, so security check (write to registry) reflects
    // the client's process, not the repl service process.
    //
    ApiStatus = NetpImpersonateClient();
    if (ApiStatus != NO_ERROR) {
        goto Cleanup;
    }
    Impersonated = TRUE;

    //
    // Check role and handle call if import half of service is not running.
    //
    ACQUIRE_LOCK_SHARED( ReplConfigLock );
    LockedConfigData = TRUE;

    if ( !ReplRoleIncludesImport( ReplConfigRole  ) ) {

        if (! ImportDirConfigDataExists( UncServerName, DirName )) {
            ApiStatus = NERR_UnknownDevDir;
            goto Cleanup;
        }

        ApiStatus = ImportDirDeleteConfigData( UncServerName, DirName );
        goto Cleanup;
    }

    //
    // Import side of service is running...
    // Get exclusive lock on client list, so we don't get confused by
    // another API thread.  This also locks the matching registry data.
    //
    ACQUIRE_LOCK( RCGlobalClientListLock );
    LockedClientList = TRUE;

    //
    // Call somebody to delete from registry.
    // This has the side-effect of doing a security check.
    //
    ApiStatus = ImportDirDeleteConfigData( UncServerName, DirName );
    if (ApiStatus != NO_ERROR) {
        goto Cleanup;   // Don't forget to release lock...
    }

    // BUGBUG: Is just removing the record from list enough or do we need
    // to notify anybody?  What if there's something for this one in the
    // delay list or work queue?

    ApiStatus = ReplRemoveClientRecForDirName( DirName );

    // BUGBUG: Are registry and service out-of-sync if ApiStatus != NO_ERROR?

Cleanup:

    if (Impersonated) {
        (VOID) NetpRevertToSelf();
    }

    if (LockedClientList) {
        RELEASE_LOCK( RCGlobalClientListLock );
    }

    if (LockedConfigData) {
        RELEASE_LOCK( ReplConfigLock );
    }

    return (ApiStatus);

}
