/****************************** Module Header ******************************\
* Module Name: libmgmt.c
*
* Copyright (c) 1985-91, Microsoft Corporation
*
* This module contains the code to manage loading and freeing libraries
* in use by USER.
*
* History:
* 02-04-91 DavidPe      Created.
\***************************************************************************/

#include "precomp.h"
#pragma hdrstop

/*
 * Number of hmodule entries in the module management table.
 */
int catomSysTableEntries;

/*
 * Array of atoms that are the fully qualified path names of each managed
 * module.
 */
ATOM aatomSysLoaded[CLIBS];

/*
 * Count of processes that have LoadModule()'d each module.
 */
int acatomSysUse[CLIBS];

/*
 * Count of hooks set into each module.
 */
int acatomSysDepends[CLIBS];

/****************************************************************************\
* SetAllWakeBits
*
* This function is used to set specified wakebits in all threads.
*
* History:
* 02-04-91  DavidPe         Ported.
\****************************************************************************/

VOID SetAllWakeBits(
    UINT wWakeBit)
{
    PTHREADINFO pti;

    for (pti = gptiFirst; pti != NULL; pti = pti->ptiNext) {
        SetWakeBit(pti, wWakeBit);
    }
}


/****************************************************************************\
* GetHmodTableIndex
*
* This routine is used to return the index of a given atom within the system
* wide hmod atom table.  If the atom is not found, an attempt to allocate a
* new table entry is made.  If the attempt fails, -1 is returned.
*
* History:
* 02-04-91  DavidPe         Ported.
\****************************************************************************/

int GetHmodTableIndex(
    LPWSTR pszLibName)
{
    int i;
    ATOM atom;

    /*
     * First look for the atomize for of this string.
     */
    atom = FindAtomW(pszLibName);

    /*
     * If we didn't find it add it.
     */
    if (atom == 0) {
        atom = AddAtomW(pszLibName);

        /*
         * If we can't add the atom we're hosed
         * so return an error.
         */
        if (atom == 0) {
            return -1;
        }
    }

    /*
     * Search for atom index
     */
    for (i = 0; i < catomSysTableEntries && aatomSysLoaded[i] != atom; i++)
        ;

    if (i == catomSysTableEntries) {

        /*
         * Find empty entry for atom
         */
        for (i = 0; i < catomSysTableEntries && aatomSysLoaded[i]; i++)
            ;

        /*
         * Check if no empty entry found
         */
        if (i == catomSysTableEntries) {
            if (i == CLIBS) {
                RIP0(ERROR_NOT_ENOUGH_MEMORY);
                return -1;
            }

            /*
             * Increase table size
             */
            catomSysTableEntries++;
        }

        /*
         * Set entry
         */
        aatomSysLoaded[i] = atom;
        acatomSysUse[i] = 0;
        acatomSysDepends[i] = 0;
    }

    return i;
}


/*****************************************************************************\
* AddHmodDependency
*
* This function merely increments the dependency count of a given hmod
* atom table index.
*
* History:
* 02-04-91  DavidPe         Ported.
\*****************************************************************************/

VOID AddHmodDependency(
    int iatom)
{
    UserAssert(iatom >= 0);
    if (iatom < catomSysTableEntries) {
        acatomSysDepends[iatom]++;
    }
}


/*****************************************************************************\
* RemoveHmodDependency
*
* This function removes a system dependency on a given index into the hmod
* atom table.  If all dependencies on the hmod have been removed (the Depends
* count  reaches zero) then the QS_SYSEXPUNGE bit is set in all message
* queues so the eventually each process will do a free module on it.
*
* History:
* 02-04-91  DavidPe         Ported.
\*****************************************************************************/

VOID RemoveHmodDependency(
    int iatom)
{

    UserAssert(iatom >= 0);
    if (iatom < catomSysTableEntries &&
        --acatomSysDepends[iatom] == 0) {

        if (acatomSysUse[iatom]) {
            SetAllWakeBits(QS_SYSEXPUNGE);
        } else {
            aatomSysLoaded[iatom] = 0;
        }
    }
}


/*****************************************************************************\
* xxxLoadHmodIndex
*
* This function attempts to load the hmodule specified by iatom into the
* system hmod table.  Updates the per-process bitmap accordingly.  Returns
* NULL on success.
*
* History:
* 02-04-91  DavidPe         Ported.
\*****************************************************************************/

HANDLE xxxLoadHmodIndex(
    int iatom)
{
    WCHAR pszLibName[MAX_PATH];
    HANDLE hmod;

    if (iatom >= catomSysTableEntries) {
        SRIP0(ERROR_INVALID_PARAMETER, "Index out of range");
        return NULL;
    }

    GetAtomNameW(aatomSysLoaded[iatom], pszLibName, sizeof(pszLibName)/sizeof(WCHAR));

    /*
     * Call back the client to load the library.
     */
    hmod = ClientLoadLibrary(pszLibName);

    if (hmod != NULL) {
        /*
         * Check to make sure another thread hasn't loaded this library
         * while we were outside the critical section.
         */
        if (!TESTHMODLOADED(iatom)) {
            /*
             * Go ahead and bump the reference count.
             */
            acatomSysUse[iatom]++;
            SETHMODLOADED(iatom, hmod);

        } else {
            /*
             * Another thread loaded it while we were outside the
             * critical section.  Unload it so the system's
             * reference count is correct.
             */
            ClientFreeLibrary(PtiCurrent()->ppi->ahmodLibLoaded[iatom]);
        }
    }

    return hmod;
}


/***********************************************************************\
* DoSysExpunge
*
* This function is called when a thread wakes up and finds its
* QS_SYSEXPUNGE wakebit set.
*
* History:
* 02-04-91  DavidPe         Ported.
\***********************************************************************/

VOID xxxDoSysExpunge(
    PTHREADINFO pti)
{
    int i;

    /*
     * Clear this first before we potentially leave the critical section.
     */
    pti->fsChangeBits &= ~QS_SYSEXPUNGE;
    pti->fsWakeBits &= ~QS_SYSEXPUNGE;

    /*
     * Scan for libraries that have been freed
     */
    for (i = 0; i < catomSysTableEntries; i++) {
        if ((acatomSysDepends[i] == 0) && (aatomSysLoaded[i] != 0) &&
                TESTHMODLOADED(i)) {

            HANDLE hmodFree = pti->ppi->ahmodLibLoaded[i];

            /*
             * Clear this hmod for this process before we leave the
             * critical section.
             */
            CLEARHMODLOADED(i);

            /*
             * Decrement the count of processes that have loaded this
             * .dll.  If there are no more, then destroy the reference
             * to this .dll.
             */
            if (--acatomSysUse[i] == 0) {
                DeleteAtom(aatomSysLoaded[i]);
                aatomSysLoaded[i] = 0;
            }

            /*
             * Call back the client to free the library...
             */
            ClientFreeLibrary(hmodFree);
        }
    }
}
