/* ---File: printman.h ----------------------------------------------------
 *
 *  Description:
 *    NT/LanMan 2.0 Print Manager Windows Application Header file which
 *    contains common definitions, constants, typedefs, external
 *    declarations, function prototypes, etc.
 *
 *    This document contains confidential/proprietary information.
 *    Copyright (c) 1990-1992 Microsoft Corporation, All Rights Reserved.
 *
 * Revision History:
 *    [00]   21-Nov-90       stevecat    created
 *    [01]   03-Jan-91       stevecat    Modified to use Windows MDI
 *    [02]   25-Mar-91       stevecat    Modified to use NT WINSPOOL APIs
 *    [03]   13-Jan-92       stevecat    New PrintMan UI
 *
 * ---------------------------------------------------------------------- */

#define MSCTLS /* Use common controls for status bars and headers */


//==========================================================================
//                             Include files
//==========================================================================
//  C Runtime
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//  Windows
/*
 * To help reduce the amount of memory required by the C compiler's
 * preprocessor pass, reduce the amount of preprocessor work as much
 * as possible. The various NO* #defines for the Windows SDK include
 * file (WINDOWS.H) help cut down on memory usage by the preprocessor.
 * In addition, the removal of the COMM functions of Windows will
 * eliminate some warnings generated by the Microsoft C Compiler at
 * warning levels 3 and 4 (due to the bitfield structures in WINDOWS.H).
 */

/* Windows 3.0 toolkit */
#define NOATOM              /* Atom Manager routines */
#define NOCLIPBOARD         /* Clipboard routines */
#define NOCOMM              /* to eliminate lint warnings in windows.h */
#define NODEFERWINDOWPOS    /* DeferWindowPos routines */
#define NOGDICAPMASKS       /* CC_*, LC_*, PC_*, CP_*, TC_*, RC_ */
#define NOKANJI             /* Kanji support stuff. */
#define NOMETAFILE          /* typedef METAFILEPICT */
#define NOMINMAX            /* No min(a,b) or max(a,b) */
#define NOOPENFILE          /* OpenFile (), OemToAnsi, AnsiToOem, and OF_* */
#define NOPROFILER          /* Profiler interface. */
#define NOSOUND             /* Sound driver routines */
#undef OEMRESOURCE          /* OEM Resource values */
#undef NOLSTRING            /* using lstrlen () */

#include <windows.h>
#include <tchar.h>
#include <wchar.h>
#include <sedapi.h>
#include <winspool.h>

//  Print Manager specific
#include "pmdef.h"
#include "setupdlg.h"

#define WIN32
#include <port1632.h>
#include <commctrl.h>

#ifdef HEAPCHECK
#include "heap.h"
#endif

//==========================================================================
//                             Debug Macros
//==========================================================================
//  NOTE:  These macros will print out the Module name, Source file name and
//         line number where the RIP occurred.
//
//  NOTE:  Use of these macros require the string "strProgram" to always be
//         defined in global data space.  (e.g.  TCHAR *strProgram = "PLOTTER";)
//
//  NOTE:  The RIP() macro requires that a Message parameter be present, but
//         RIP("") and RIP(NULL) are valid if no message is required.
//
#if DBG

/* Quick fix:
 *
 * Ensure DbgPrint and DbgBreakPoint are prototyped,
 * so that we're not screwed by STDCALL.
 */
VOID
DbgBreakPoint(
    VOID
    );

VOID DbgMsg( CHAR *MsgFormat, ... );


#if DBG
#define GLOBAL_DEBUG_FLAGS Debug
#endif

#define DBG_NONE       0x00000000
#define DBG_INFO       0x00000001
#define DBG_WARNING    0x00000002
#define DBG_ERROR      0x00000004
#define DBG_TRACE      0x00000008
#define DBG_THREADS    0x00000010
#define DBG_ALLOC      0x00000020
#define DBG_MUTEX      0x00000040

/* These flags are not used as arguments to the DBGMSG macro.
 * You have to set the high word of the global variable to cause it to break.
 * It is ignored if used with DBGMSG.
 * (Here mainly for explanatory purposes.)
 */
#define DBG_BREAK_ON_WARNING    ( DBG_WARNING << 16 )
#define DBG_BREAK_ON_ERROR      ( DBG_ERROR << 16 )

/* Double braces are needed for this one, e.g.:
 *
 *     DBGMSG( DBG_ERROR, ( "Error code %d", Error ) );
 *
 * This is because we can't use variable parameter lists in macros.
 * The statement gets pre-processed to a semi-colon in non-debug mode.
 *
 * Set the global variable GLOBAL_DEBUG_FLAGS via the debugger.
 * Setting the flag in the low word causes that level to be printed;
 * setting the high word causes a break into the debugger.
 * E.g. setting it to 0x00040006 will print out all warning and error
 * messages, and break on errors.
 */
#define DBGMSG( Level, MsgAndArgs ) \
{                                   \
    if( ( Level & 0xFFFF ) & GLOBAL_DEBUG_FLAGS ) \
        DbgMsg MsgAndArgs;      \
    if( ( Level << 16 ) & GLOBAL_DEBUG_FLAGS ) \
        DbgBreakPoint(); \
}

#else
#define DBGMSG( level, args )
#endif //  DBG


//
//  This define is used to turn on extra error reporting for local testing
//
#define TESTING TRUE

//==========================================================================
//                             Definitions
//==========================================================================
typedef CREATESTRUCT    PORTCREATESTRUCT;
typedef WNDCLASS        PORTWNDCLASS;

#define NOQUEUEFIELDS   4
#define NOJOBFIELDS     6


//
//  Extra fields on Win struct to hold info in MDI window
//
#define GWL_PMDIWIN         0       //  Ptr to QUEUE struct for this window

// IDM_ constants for menu IDs
// Note these double as IDs for the corresponding help messages

// Printer Menu

#define IDM_PRINTER             1100
#define IDM_CONNECTTOPRINTER    1101
#define IDM_REMOVECONNECTION    1102  /* This menu item no longer exists,   */
                                      /* but the toolbar button does, and   */
                                      /* there is help associated with it.  */
#define IDM_PROPERTIES          1103
#define IDM_PRINTER_PAUSE       1104
#define IDM_PRINTER_RESUME      1105
#define IDM_FORMS               1106
#define IDM_INSTALLPRINTER      1107
#define IDM_DELETEPRINTER       1108
#define IDM_PURGEPRINTER        1109
#define IDM_SERVERVIEWER        1110
#define IDM_EXIT                1111
#define IDM_RETURN              1112  // used for de-minimizing child window

// Document Menu

#define IDM_DOCUMENT            1200
#define IDM_REMOVEDOC           1201
#define IDM_DOCTAILS            1202
#define IDM_DOCUMENT_PAUSE      1203
#define IDM_DOCUMENT_RESUME     1204
#define IDM_RESTART             1205

// Options Menu

#define IDM_OPTIONS             1300
#define IDM_FONT                1301
#define IDM_REFRESHRATE         1302
#define IDM_TOOLBAR             1303
#define IDM_STATUSBAR           1304
#define IDM_SAVESETTINGS        1305

// Security Menu

#define IDM_SECURITY            1400
#define IDM_PERMISSIONS         1401
#define IDM_AUDITING            1402
#define IDM_OWNER               1403

// Window Menu

#define IDM_WINDOW              1500
#define IDM_CASCADE             1501
#define IDM_TILEHORZ            1502
#define IDM_TILEVERT            1503
#define IDM_ARRANGE             1504
#define IDM_REFRESH             1505
#define IDM_CLOSE               1506
#define IDM_CLOSEALL            1507

// Help Menu

#define IDM_HELP                1600
#define IDM_HELP_CONTENTS       1601
#define IDM_HELP_SEARCH         1602
#define IDM_HELP_HOWTOUSE       1603
#define IDM_ABOUT               1604

// Tab command for accelerator table.
// This causes tab to be translated to a command.
// This ID must not coincide with a menu ID.

#define IDC_TAB                 1650

// IDs for extra help messages for popups and menu items that don't have IDs.

#define IDS_HELPFIRSTPOPUP      1700
#define IDS_HELPPRINTER         1700 // These five must be consecutive
#define IDS_HELPDOCUMENT        1701
#define IDS_HELPOPTIONS         1702
#define IDS_HELPSECURITY        1703
#define IDS_HELPWINDOW          1704
#define IDS_HELPHELP            1705

#define IDS_HELP_MDIWINDOW      1706
#define IDS_HELP_MOVINGDOCUMENT 1707

#define IDM_FIRSTCHILD         1000

// Resource file string table section entry ids

#define  IDS_PRINTMANAGER                   600
#define  IDS_MESSAGE_TITLE                  IDS_PRINTMANAGER
#define  IDS_COULDNOTOPENPRINTER            601
#define  IDS_CONFIRMDELETE                  602
#define  IDS_PRINTER                        603
#define  IDS_DEFAULT                        604
#define  IDS_DEFAULTLABEL                   605
#define  IDS_UNTITLED                       606
#define  IDS_CREATEPRINTER                  610
#define  IDS_OTHER                          611
#define  IDS_UNNAMED                        612
#define  IDS_PRINTERDETAILS                 613
#define  IDS_SELECTSEPARATORPAGE            614
#define  IDS_SEPARATORFILES                 615
#define  IDS_SEPARATORFILTER                616
#define  IDS_ALLFILES                       617
#define  IDS_ALLFILTER                      618
#define  IDS_COULDNOTCREATEPRINTER          621
#define  IDS_COULDNOTSETPRINTER             622
#define  IDS_COULDNOTDELETEPRINTER          623
#define  IDS_COULDNOTPAUSEPRINTER           624
#define  IDS_COULDNOTRESUMEPRINTING         625
#define  IDS_COULDNOTPURGEPRINTER           626
#define  IDS_COULDNOTPAUSEDOCUMENT          627
#define  IDS_COULDNOTRESUMEDOCUMENT         628
#define  IDS_COULDNOTLOADLIBRARY            629
#define  IDS_COULDNOTFINDPROCEDURE          630
#define  IDS_PRINT                          632
#define  IDS_ADMINISTER                     633
#define  IDS_ADMINISTERDOCUMENTS            634
#define  IDS_NOACCESS                       635
#define  IDS_COULDNOTENUMERATEPRINTERS      636
#define  IDS_COULDNOTFINDINFFILE            637
#define  IDS_DELETEALLPRINTJOBS_S           639
#define  IDS_SERVERUNAVAILABLE              641
#define  IDS_SERVERUNAVAILABLEONSTART       642
#define  IDS_MUSTSUPPLYVALIDNAME            643
#define  IDS_COULDNOTADDPORT                644
#define  IDS_COULDNOTCONNECTTOPRINTER       645
#define  IDS_SERVERVIEWERTITLE              646
#define  IDS_COULDNOTOPENSERVERREFRESH      648
#define  IDS_COULDNOTGETSERVERINFO          649
#define  IDS_COULDNOTSHOWHELP               650
#define  IDS_COULDNOTREMOVEDOCUMENT         652
#define  IDS_FORMSONTHISCOMPUTER            653
#define  IDS_COULDNOTADDFORM                654
#define  IDS_COULDNOTDELETEFORM             655
#define  IDS_COULDNOTSETFORM                656

#define  IDS_INSUFFPRIV_DELETEPRINTER       661
#define  IDS_INSUFFPRIV_PAUSEPRINTER        662
#define  IDS_INSUFFPRIV_RESUMEPRINTING      663
#define  IDS_INSUFFPRIV_PURGEPRINTER        664
#define  IDS_INSUFFPRIV_PAUSEDOCUMENT       665
#define  IDS_INSUFFPRIV_RESUMEDOCUMENT      666
#define  IDS_INSUFFPRIV_DELETEDOCUMENT      667
#define  IDS_INSUFFPRIV_ADDPORT             668
#define  IDS_INSUFFPRIV_DELETEPORT          669
#define  IDS_INSUFFPRIV_SECURITY            670
#define  IDS_ERRORRUNNINGSETUP              672
#define  IDS_COULDNOTREMOVECONNECTION       673
#define  IDS_COULDNOTDELETEPORT             675
#define  IDS_CONFIRMDELETEREMOTE            676
#define  IDS_CONFIRMDELETECONNECTION        677
#define  IDS_INCHES                         678
#define  IDS_CENTIMETERS                    679
#define  IDS_BYTES                          680
#define  IDS_SHARENAMEMAYBETOOLONG          681
#define  IDS_COULDNOTUPDATESECURITY         684
#define  IDS_PERMISSIONNOLONGERGRANTED      685
#define  IDS_FAILUREREOPENINGPRINTER        686
#define  IDS_INSUFFPRIV_RESTARTDOCUMENT     687
#define  IDS_COULDNOTRESTARTDOCUMENT        688
#define  IDS_CANNOTGETSECURITYINFO          689
#define  IDS_ACCESSDENIED                   690
#define  IDS_UNKNOWN_ERROR                  691
#define  IDS_PERMISSIONS_EDITOR_FAILED      692
#define  IDS_AUDIT_DIALOG_FAILED            693
#define  IDS_TAKE_OWNERSHIP_FAILED          694
#define  IDS_COULDNOTSETDEVMODE             695
//  #define  IDS_ADD_PORT                       696
#define  IDS_CANNOT_COPY_DRIVER_FILES       697
#define  IDS_ERROR_VALIDATING_ACCESS        698
#define  IDS_NO_PRINT_PROCESSORS            699
#define  IDS_GETTING_PRINTER_INFO           700
#define  IDS_SAVE_SETTINGS                  701
#define  IDS_AUDIT_PRINT                    702
#define  IDS_AUDIT_ADMINISTER               703
#define  IDS_AUDIT_DELETE                   704
#define  IDS_CHANGE_PERMISSIONS             705
#define  IDS_TAKE_OWNERSHIP                 706
#define  IDS_INSUFF_PRIV_AUDITING           707
#define  IDS_COULD_NOT_VIEW_SERVER          708
#define  IDS_INVALID_PRINTER_STATE          709
//                                          710
#define  IDS_OPEN_PRINTER_FAILED            711
#define  IDS_INSUFFPRIV_DOCUMENT_SETTINGS   712
#define  IDS_COULDNOTSETDOCUMENT            713
#define  IDS_COULDNOTSETDOCUMENTTIMES       714
#define  IDS_COULD_NOT_REORDER_JOB          715
#define  IDS_NO_DRIVERS_INSTALLED           716
#define  IDS_NO_DRIVER_ON_SERVER            717
#define  IDS_PRINTER_PROPERTIES_FAILED      718
#define  IDS_PRINTER_NOT_LOADED             719

//
// For SetupDlg support:
//
#define  IDS_PRINTER_TYPE                   800
#define  IDS_MONITOR_TYPE                   801

#define  IDS_PRINTER_INSTALL_TITLE          810
#define  IDS_MONITOR_INSTALL_TITLE          811

#define  IDS_PRINTER_SELECT_TITLE          820
#define  IDS_MONITOR_SELECT_TITLE          821


#define IDS_FIRST_HEADER_PRINTER            160       //  Job display Column Header titles
#define IDS_FIRST_HEADER_SERVER             170       //  Job display Column Header titles
#define STATUS                  192       //  Status messages
#define IDS_PAUSEDON_S          192
#define IDS_PRINTING_S          193
#define IDS_ERRORPRINTING_S     194
#define IDS_NOJOBSTOPRINT       195
#define IDS_PAGESCOMPLETED_D    196
#define IDS_DOCUMENTSQUEUED_D   197

#define IDS_PAGESCOMPLETED_D    196
#define IDS_DOCUMENTSQUEUED_D   197

#define IDS_PAUSED              213
#define IDS_ERROR               214
#define IDS_PENDING_DELETION    215
#define IDS_READY               216

#define IDS_OFFLINE             220
#define IDS_PAPEROUT            221
#define IDS_DELETING            222
#define IDS_SPOOLING            223
#define IDS_PRINTING            224
#define IDS_NOSTATUS            225
#define IDS_ACCESS_DENIED       226
#define IDS_STATUS_UNKNOWN      227
#define IDS_ERROR_N             228
#define IDS_STATUS_LOADING      229
#define IDS_PRINTED             230
#define IDS_PRINTANDSPOOL       231

#define IDS_PAPER_JAM           232
#define IDS_MANUAL_FEED         233
#define IDS_IO_ACTIVE           234
#define IDS_BUSY                235
#define IDS_OUTPUT_BIN_FULL     236
#define IDS_NOT_AVAILABLE       237
#define IDS_WAITING             238
#define IDS_PROCESSING          239
#define IDS_INITIALIZING        240
#define IDS_WARMING_UP          241
#define IDS_TONER_LOW           242
#define IDS_NO_TONER            243
#define IDS_PAGE_PUNT           244
#define IDS_USER_INTERVENTION   245
#define IDS_OUT_OF_MEMORY       246
#define IDS_DOOR_OPEN           247
#define IDS_PAPER_PROBLEM       248

/* Child window control IDs
 */
// Button IDs
#define IDB_PAUSERESUME     1
#define IDB_DELETE          2

// Other IDs
#define ID_OBJLIST          7   // List box in MDI windows
#ifdef MSCTLS
#define ID_STATUSBAR        8   // Frame window status bar
#define ID_HEADER           9   // MDI movable headers
#define ID_TOOLBAR         10   // Toolbar ID
#endif /* MSCTLS */


/* Icons:
 */
#define IDI_PRINTMAN        1
#define IDI_SERVER          2
#define IDI_CONNECT         3
#define IDI_SHARED          4

#define IDC_REORDER         5


//  Toolbar combo-box constants

#define IDTB_CB_PRINTERS       5        // Toolbar Combobox control ID
#define MAX_PRINTER_NAME_LEN  32        // Toolbar combobox max string length
#define MAX_PRINTER_DESC_LEN  64
#define MAX_SHARE_NAME_LEN    12

#define K           1024                // Kilobyte
#define TBZ         256                 // Text output buffer size
#define NBZ         10*K                // Network API buffer size

/* These values are used in the LocalAlloc() call in InitInstance() routine
 * to get queue names and nicknames from .INI file.
 */
#define KEYBZ       1*K                 // Keyname Buffer size
#define VALBZ       256                 // Value Buffer size (on stack)

#define NOSELECTION (DWORD)(-1)         // No Print Job selected index value

#define DYTOOLBAR    27

//
//  Toolbar ICON bitmap stuff

#define BITMAPS             100
#define TOOLBAR             101
#define FILES_WIDTH         16
#define FILES_HEIGHT        16


/* Indexes of bitmaps for Server and Printer viewer status */
#define BM_IND_STATUS_XBASE         0
#define BM_IND_STATUS_YBASE         0
#define BM_IND_PRINTER_READY        0
#define BM_IND_PRINTER_PAUSED       1
#define BM_IND_PRINTER_ERROR        2
#define BM_IND_DOCUMENT_PRINTING    3
#define BM_IND_DOCUMENT_PAUSED      4
#define BM_IND_DOCUMENT_ERROR       5


#define MSG_ERROR           MB_OK | MB_ICONSTOP
#define MSG_WARNING         MB_OK | MB_ICONEXCLAMATION
#define MSG_INFORMATION     MB_OK | MB_ICONINFORMATION
#define MSG_YESNO           MB_YESNO | MB_ICONQUESTION
#define MSG_CONFIRMATION    MB_OKCANCEL | MB_ICONEXCLAMATION

#define MDIWIN_PRINTER          0
#define MDIWIN_LOCALPRINTER     0
#define MDIWIN_SERVER           1
#define MDIWIN_NETWORKPRINTER   2
#define MDIWIN_LOCALNETWORKPRINTER  3   /* Local printer which looks like a network printer */

/* Datatypes used in printer and server lists:
 */
#define MDIDATA_DWORD                   0
#define MDIDATA_DWORD_BLANK_IF_ZERO     1
#define MDIDATA_PSZ                     2
#define MDIDATA_PSZ_UNTITLED_IF_NULL    3
#define MDIDATA_TIME                    4
#define MDIDATA_SIZE                    5
#define MDIDATA_PRINTER_STATUS          6
#define MDIDATA_JOB_STATUS              7
#define MDIDATA_ICON                    8

/* Indices of column headings for the job list for each printer:
 */
#define MDIHEAD_JOB_STATUS    0
#define MDIHEAD_JOB_DOCNAME   1
#define MDIHEAD_JOB_OWNER     2
#define MDIHEAD_JOB_PRINTEDAT 3
#define MDIHEAD_JOB_PAGES     4
#define MDIHEAD_JOB_SIZE      5
#define MDIHEAD_JOB_PRIORITY  6
#define MDIHEAD_JOB_COUNT     7

/* Indices of column headings for the printer list for each server:
 */
#define MDIHEAD_PRINTER_PRINTER    0
#define MDIHEAD_PRINTER_STATUS     1
#define MDIHEAD_PRINTER_JOBS       2
#define MDIHEAD_PRINTER_PORT       3
#define MDIHEAD_PRINTER_TYPE       4
#define MDIHEAD_PRINTER_COUNT      5

/* This should not clash with any of the PRINTER_STATUS_* values in winspool.h:
 */
#define PRINTER_STATUS_UNKNOWN         0x80000000
#define PRINTER_STATUS_ACCESS_DENIED   0x40000000
#define PRINTER_STATUS_LOADING         0x20000000


/* The next value is the largest number of headers in any MDI window:
 */
#define MAX_HEADERS   MDIHEAD_JOB_COUNT

/* Space for 16x16 status bitmaps:
 */
#define STATUS_BITMAP_WIDTH     16
#define STATUS_BITMAP_HEIGHT    16
#define STATUS_BITMAP_MARGIN     4  /* (either side) */
#define STATUS_BITMAP_SPACE     ( STATUS_BITMAP_WIDTH + ( 2 * STATUS_BITMAP_MARGIN ) )

/* These are the push-button bitmap widths:
 */
#define DXBUTTON            24
#define DYBUTTON            22

// #ifdef JAPAN
#define RESOURCE_STRING_LENGTH  160
// #else
// #define RESOURCE_STRING_LENGTH  80
// endif

#define TIME_STRING_LENGTH      20
#define ENTRYFIELD_LENGTH      256

#define REPAINT_NONE             0
#define REPAINT_IF_CHANGED       1
#define REPAINT_FORCE            2

/* International time stuff:
 */
#define TIMESEP_LENGTH  2
#define AM              0
#define PM              1
#define AMPM_LENGTH     9


/* For OpenPrinterForSpecifiedAccess
 * (this must not clash with PRINTER_ACCESS_ADMINISTER,
 * PRINTER_ACCESS_USE or READ_CONTROL):
 */
#define PRINTER_ACCESS_HIGHEST_PERMITTED    0x41ACCE55
#define PRINTER_ACCESS_DENIED               0x00000000

/* Print Manager private messages:
 */
#define WM_STATUS_CHANGED   WM_USER
#define WM_UPDATE_STATUS    WM_USER+1
#define WM_UPDATE_LIST      WM_USER+2
#define WM_UPDATE_DEFAULT   WM_USER+3
#define WM_PRINTER_ADDED    WM_USER+4
#define WM_REFRESH_ERROR    WM_USER+5
#define WM_THREAD_ERROR     WM_USER+6
#define WM_DELETE_PRINTER   WM_USER+7

/* Define some constants to make parameters to CreateEvent a tad less obscure:
 */
#define EVENT_RESET_MANUAL                  TRUE
#define EVENT_RESET_AUTOMATIC               FALSE
#define EVENT_INITIAL_STATE_SIGNALED        TRUE
#define EVENT_INITIAL_STATE_NOT_SIGNALED    FALSE

#define CREATE_PRINTER_SHARED   0x00000001
#define CREATE_PRINTER_ICONIC   0x00000002

#define ALL_FLAGS_ARE_SET( Flags, Mask ) ( ( (Flags) & (Mask) ) == (Mask) )


//==========================================================================
//                                Macros
//==========================================================================
#define WPARAM_ID(wParam)               LOWORD(wParam)

#define ZERO_OUT( pStructure )  memset( (pStructure), 0, sizeof( *(pStructure) ) )

#define GETPRINTMAN_DATA( hwnd ) \
    (PPRINTMAN_DATA)GetWindowLong( hwnd, GWL_USERDATA )

#define SETDLGITEMFONT(hwnd, id, hfont) \
    SendDlgItemMessage(hwnd, id, WM_SETFONT, (WPARAM)hfont, 0L)

#define ADDCOMBOSTRING(hwnd, id, string) \
    SendDlgItemMessage(hwnd, id, CB_ADDSTRING, 0, (LONG)string)

#define INSERTCOMBOSTRING(hwnd, id, i, string) \
    SendDlgItemMessage(hwnd, id, CB_INSERTSTRING, (WPARAM)i, (LONG)string)

#define SETCOMBOSELECT(hwnd, id, i) \
    SendDlgItemMessage(hwnd, id, CB_SETCURSEL, (WPARAM)i, 0L)

#define GETCOMBOSELECT(hwnd, id) \
    SendDlgItemMessage(hwnd, id, CB_GETCURSEL, 0, 0L)

#define GETCOMBOTEXT(hwnd, id, i, string) \
    SendDlgItemMessage(hwnd, id, CB_GETLBTEXT, (WPARAM)i, (LONG)string)

#define GETCOMBOSELECTTEXT(hwnd, id, string) \
    GETCOMBOTEXT(hwnd, id, GETCOMBOSELECT(hwnd, id), string)

#define FINDCOMBOSTRING(hwnd, id, string) \
    SendDlgItemMessage(hwnd, id, CB_FINDSTRING, -1, (LONG)string)

#define RESETLIST(hwnd, id) \
    SendDlgItemMessage(hwnd, id, LB_RESETCONTENT, 0, 0L);

#define SETLISTSELECT(hwnd, id, i) \
    SendDlgItemMessage(hwnd, id, LB_SETCURSEL, (WPARAM)i, 0L)

#define GETLISTSELECT(hwnd, id) \
    SendDlgItemMessage(hwnd, id, LB_GETCURSEL, 0, 0L )

#define GETLISTTEXT(hwnd, id, i, string) \
    SendDlgItemMessage(hwnd, id, LB_GETTEXT, (WPARAM)i, (LONG)string)

#define GETLISTSELECTTEXT(hwnd, id, string) \
    GETLISTTEXT(hwnd, id, GETLISTSELECT(hwnd, id), string)


#define IS_BUTTON_ENABLED(idCommand) (BOOL)LOWORD(SendMessage( hwndToolbar, \
                                                               TB_ISBUTTONENABLED, \
                                                               idCommand,\
                                                               0L ))


#define SEND_THREAD_MESSAGE( hWnd, msg, wparam, lparam )    \
    WaitForSingleObject( ThreadMessageRead, INFINITE ),     \
    ThreadMessage.hwnd = (hWnd),                            \
    ThreadMessage.message = (msg),                          \
    ThreadMessage.wParam = (WPARAM)(wparam),                \
    ThreadMessage.lParam = (LPARAM)(lparam),                \
    SetEvent( ThreadMessageWritten )

#define DISPATCH_THREAD_MESSAGE( pMsg ) \
    PostMessage( (pMsg)->hwnd, (pMsg)->message, (pMsg)->wParam, (pMsg)->lParam ),   \
    SetEvent( ThreadMessageRead )


/* ENUM_* macros designed so that spooler Enum* calls can be made
 * without worrying about reallocating buffers when required.
 * They call EnumGeneric, which does the appropriate reallocation,
 * making it completely transparent to the caller,
 * as they take exactly the same parameters as the corresponding
 * spooler API.
 * Any buffer passed in must have been allocated via AllocSplMem.
 */
#define ENUM_PRINTERS( Flags, Name, Level, pPrinterEnum, cbBuf, pcbNeeded, pcReturned ) \
    EnumGeneric( (PROC)EnumPrinters,        \
                 (Level),                   \
                 (LPBYTE *)&(pPrinterEnum), \
                 (cbBuf),                   \
                 (pcbNeeded),               \
                 (pcReturned),              \
                 (PVOID)(Flags),            \
                 (Name),                    \
                 NULL )

#define ENUM_JOBS( hPrinter, FirstJob, NoJobs, Level, pJob, cbBuf, pcbNeeded, pcReturned )  \
    EnumGeneric( (PROC)EnumJobs,            \
                 (Level),                   \
                 (LPBYTE *)&(pJob),         \
                 (cbBuf),                   \
                 (pcbNeeded),               \
                 (pcReturned),              \
                 (PVOID)(hPrinter),         \
                 (PVOID)(FirstJob),         \
                 (PVOID)(NoJobs) )

#define ENUM_PORTS( pName, Level, pPorts, cbBuf, pcbNeeded, pcReturned )    \
    EnumGeneric( (PROC)EnumPorts,           \
                 (Level),                   \
                 (PBYTE *)&(pPorts),        \
                 (cbBuf),                   \
                 (pcbNeeded),               \
                 (pcReturned),              \
                 (pName),                   \
                 NULL,                      \
                 NULL )

//==========================================================================
//                   Typedefs and Structure Definitions
//==========================================================================

//**************************************************************************
//
//  QUEUE is a linked-list structure containing printer information
//
//**************************************************************************
//  NOTE:
//

/* SETUP_DATA
 *
 * This is allocated in Printer Properties dialog
 * for installable printer drivers read in from the PRINTER.INI file.
 */
typedef struct _SETUP_DATA
{
    PTCHAR pOption;
//  PTCHAR pOptionText;
}
SETUP_DATA, *PSETUP_DATA;


typedef struct _PRINTMAN_DATA
{
    PVOID Unused;
}
PRINTMAN_DATA, *PPRINTMAN_DATA;


typedef struct _COLUMN
{
    INT   Width;
    PTCHAR Text;
    INT   Datatype;      // Type of data displayed in this column
    INT   Offset;        // Offset of the data from pData in MDIWIN_INFO
}
COLUMN, *PCOLUMN;

typedef int  (WINAPI *REFRESHPROC)();             // new 32-bit version
typedef VOID (WINAPI *INITTHREADPROC)();
typedef BOOL (WINAPI *CHECKQUITPROC)();

typedef struct _MDIWIN_INFO
{
/* 00 */ DWORD    WindowType;         //
/* 04 */ HWND     hwndList;           //   Queue/Job list box Window
/* 08 */ HWND     hwndHeader;
/* 0c */ DWORD    TopIndex;
/* 10 */ DWORD    cNumLines;          //   Number of lines in client area
/* 14 */ PBYTE   *ppData;             //   Generic data pointer
/* 18 */ int      DataSize;           //   E.g. sizeof JOB_INFO_2
/* 1c */ PDWORD   pcObjects;          //   E.g. how many printers or jobs
/* 20 */ int      cColumns;           //   Number of headings
/* 24 */ PCOLUMN  pColumns;           //   Heading positions
/* 28 */ int      IconStatus;         //   Offset into *ppData for status icon
/* 2c */ DWORD    ObjSelected;
/* 30 */ PDWORD   pSelObjId;
/* 34 */ DWORD    DragPosition;       //   Used for reordering print jobs in Printer
/* 38 */ DWORD    PrevSelection;      //   For toggling selection with space bar
/* 3c */ PBYTE   *ppSelData;          //   Generic data pointer
/* 40 */ DWORD    IdOffset;
/* 44 */ PDWORD   pFirstEnumObj;
/* 48 */ PDWORD   pcEnumObjs;
/* 4c */ REFRESHPROC pfnRefresh;      //   Call this when we get a refresh request
/* 50 */ CHECKQUITPROC pfnCheckQuit;
/* 54 */ INITTHREADPROC pfnInitThread; //   Called at beginning of worker thread, can be NULL
/* 58 */ HANDLE   RefreshSignal;      //   Event is signaled if refresh allowed
/* 5c */ HICON    hicon;
/* 60 */ PVOID    pContext;           //   pQueue etc.
/* 64 */ HWND     hwnd;
/* 68 */ BOOL     Alive;
/* 6c */ PHANDLE  phWaitObject;       //   hPrinter or hServer
/* 70 */ DWORD    ThreadId;
/* 74 */ DWORD    WaitFlags;          //   Changes to wait for
/* 78 */ DWORD    Changes;            //   Changes that happened
/* 7c */ HANDLE   DataMutex;
/* 80 */ DWORD    Status;
#ifdef SEP_WAITHANDLE
         PHANDLE  phMain;
#endif
#if DBG
/* 84 */ DWORD    DataMutexOwner;
/* 88 */ DWORD    DataMutexCount;
#endif /* DBG */
} MDIWIN_INFO, *PMDIWIN_INFO;

typedef struct _QUEUE {             // que
/* 00 */ DWORD            AccessGranted; // Administer, User or READ_CONTROL
/* 04 */ LPTSTR            pServerName;
/* 08 */ LPTSTR            pPrinterName;
/* 0c */ HANDLE           hPrinter;
/* 10 */ DWORD            cbPrinterBuf;  // Size of LPPRINTER struct buffer
/* 14 */ LPPRINTER_INFO_2 pPrinter;      // Spooler PRINTER structure pointer
/* 18 */ DWORD            cbJobsBuf;     // Size of LPJOB buffer
/* 1c */ LPJOB_INFO_2     pJobs;         // Print Job array pointer
/* 20 */ DWORD            cJobs;         // Actual Count of Jobs in pJobs buffer
/* 24 */ DWORD            SelJobId;      // Job Id of selected print job
/* 28 */ DWORD            cbSelJob;
/* 2c */ LPJOB_INFO_2     pSelJob;
/* 30 */ DWORD            FirstEnumJob;
/* 34 */ DWORD            cEnumJobs;
/* 38 */ DWORD            Error;
/* 3c */ PMDIWIN_INFO     pMDIWinInfo;
#ifdef SEP_WAITHANDLE
         HANDLE           hPrinterWait;
#endif
#if 0
/* 40 */ HANDLE           WaitForInit;   // Event signaled or NULL if initialization complete
#endif
} QUEUE;

typedef struct _SERVER_CONTEXT
{
/* 00 */ DWORD            AccessGranted; // Administer, User or READ_CONTROL
/* 04 */ LPTSTR            pServerName;
/* 08 */ HANDLE           hServer;
/* 0c */ LPPRINTER_INFO_2 pPrinters;
/* 10 */ DWORD            cPrinters;
/* 14 */ DWORD            cbPrinters;
/* 18 */ DWORD            SelPrinterId;
/* 1c */ LPPRINTER_INFO_2 pSelPrinter;
/* 20 */ DWORD            FirstEnumPrinter;
/* 24 */ DWORD            cEnumPrinters;
/* 28 */ DWORD            Error;
/* 2c */ PMDIWIN_INFO     pMDIWinInfo;
         HANDLE           hServerWait;
}
SERVER_CONTEXT, *PSERVER_CONTEXT;

/*

  Each member of a PRT_PROP_DRIVER array points to the translatable name
  of a printer driver.  If it is already installed the Installed flag
  is TRUE, and Index is an index into an array of DRIVER_INFO_1 structures.
  If it is not installed, Installed is FALSE, and Index specifies a
  SETUP_DATA element in the linked list.

  The idea behind this is that it is now possible to sort the driver names
  alphabetically on pName regardless of whether the driver is installed,
  thus making the distinction more or less transparent to the user.
  The only difference is that the SETUP utility will be invoked
  if the user selects a non-installed driver then presses OK in the
  Printer Properties or Create Printer dialog.


  +-----------------+           +-----------------+
  | PRT_PROP_DRIVER |           | PRT_PROP_DRIVER |
  +-+-----------+---+           +-+-----------+---+
 +--| pName                   +--| pName     
 |  +-----------+              |  +-----------+
 |  | Installed | TRUE         |  | Installed | FALSE
 |  +-----------+              |  +-----------+
 |  | Index     | M            |  | Index     | N
 |  +-----------+              |  +-----------+
 |                             |
 |                             |
 |  Mth element in             |  Nth element in        Buffer of strings
 |  array returned by          |  linked list           returned by
 |  EnumPrinterDrivers:        |  of options and text:  GetOptionList:
 |  +---------------+          |  +------------+        +--------------------+
 |  | DRIVER_INFO_1 |          |  | SETUP_DATA |        | "Printer N-1"      |
 |  +-+-------+-----+          |  +-+----------+--+     +--------------------+
 | +--| pName |                |    | pOption     |---->| "My Laser Printer" |
 | |  +-------+                |    +-------------+     +--------------------+
 | |                           | +--| pOptionText |     | "Printer N+1"      |
 | |                           | |  +-------------+     +--------------------+
 | |                           | |  | pNext       |     | "Printer N+2"      |
 | |                           | |  +-------------+     +--------------------+
 | |                           | |
 | |                           | |
 | |                           | |
 +-+-> "IQ Spottyjet Printer"  +-+-> "Mein Laser-Drucker"

                                     (translatable)

*/

typedef struct _PRT_PROP_DRIVER
{
    PTCHAR               pName;
    BOOL                Installed;
    DWORD               Index;   /* of LPPRINTER_INFO_2 if Installed, */
}                                /* otherwise PSETUP_DATA             */
PRT_PROP_DRIVERS, *PPRT_PROP_DRIVER;

typedef struct _PRT_PROP_DLG_DATA
{
    DWORD                   AccessGranted;
    LPPRINTER_INFO_2        pPrinter;
    LPPRINTER_INFO_2        pNewPrinter;
    LPTSTR                  pPrinterName;
    LPTSTR                  pServerName;
    PMDIWIN_INFO            pMDIWinInfo;
    HANDLE                  hPrinter;
    BOOL                    DriverChanged;
    LPPRINTPROCESSOR_INFO_1 pPrintProcessors;
    DWORD                   cbPrintProcessors;
    DWORD                   cPrintProcessors;
    DWORD                   PrintProcessorSelected;
    LPDATATYPES_INFO_1      pDatatypes;
    DWORD                   cbDatatypes;
    DWORD                   cDatatypes;
    DWORD                   DatatypeSelected;
    BOOL                    DetailsUpdated;
    BOOL                    PortChanged;
    DWORD                   DriverSelected;
    DWORD                   PortSelected;
    LPTSTR                  pAdditionalPorts;
    BOOL                    PrinterShared;
    DWORD                   ServerAccessGranted;
    HANDLE                  hPrtPropDlgWnd;

    INFPARMS                InfParms;

} PRT_PROP_DLG_DATA, *PPRT_PROP_DLG_DATA;

typedef struct _FORMS_DLG_DATA
{
    DWORD        AccessGranted;
    LPTSTR       pServerName;
    HANDLE       hPrinter;
    PFORM_INFO_1 pFormInfo;
    DWORD        cForms;
    BOOL         Units;  /* TRUE == metric */
} FORMS_DLG_DATA, *PFORMS_DLG_DATA;

typedef struct _SEL_MON_DLG_DATA
{
    INFPARMS     InfParms;
} SEL_MON_DLG_DATA, *PSEL_MON_DLG_DATA;


/* registry.c */

/* REGISTRY_ENTRY
 *
 * Passed to WriteRegistryData and ReadRegistryData.
 * It contains information about the data to be written to
 * or read from the registry contained in the buffer pointed at
 * by the pData parameter.
 */
typedef struct _REGISTRY_ENTRY
{
    DWORD Type;     /* Type of data (passed to RegSetValueEx) */
    DWORD Size;     /* Size of data (passed to RegSetValueEx) */
}
REGISTRY_ENTRY, *PREGISTRY_ENTRY;


typedef HANDLE  HQUEUE;
typedef QUEUE * PQUEUE;
typedef QUEUE FAR * LPQUEUE;

//**************************************************************************
//
//  SPIN button control structs and definitions
//
//**************************************************************************

#ifndef NOARROWS
typedef struct
  {
    short lineup;             /* lineup/down, pageup/down are relative */
    short linedown;           /* changes.  top/bottom and the thumb    */
    short pageup;             /* elements are absolute locations, with */
    short pagedown;           /* top & bottom used as limits.          */
    short top;
    short bottom;
    short thumbpos;
    short thumbtrack;
    BOOL  wrap;               /* allow wraparound?                     */
    BYTE  flags;              /* flags set on return                   */
  } ARROWVSCROLL;
typedef ARROWVSCROLL FAR      *LPARROWVSCROLL;

#define UNKNOWNCOMMAND 1
#define OVERFLOW       2
#define UNDERFLOW      4

#endif

//==========================================================================
//                       External Data Declarations
//==========================================================================

#if DBG
extern DWORD GLOBAL_DEBUG_FLAGS;
#endif

extern TCHAR szPrintManagerClass[];
extern TCHAR szMenuClass[];
extern TCHAR szLocalPrinterClass[];
extern TCHAR szNetworkPrinterClass[];
extern TCHAR szServerWindowClass[];

extern HANDLE   hHeap;
extern HANDLE   hInst;
extern HANDLE   hRes;
extern HWND     hwndClient;
extern HWND     hwndToolbar;
extern HWND     hwndFrame;
extern HWND     hwndPrinterList;

extern UINT     idTimer;

extern TCHAR     *strProgram;
extern TCHAR     szNtLanMan[];
extern CHAR      szSedDiscretionaryAclEditor[];

extern TCHAR     strUntitled[20];
extern TCHAR     strStatusName[128];
extern TCHAR     strStatusStatus[128];
extern TCHAR     strStatusWaiting[128];
extern TCHAR     strStatusHelp[128];

extern int      dxDefaultLabel;
extern int      dyDefaultLabel;
extern int      dyDriveItem;
extern int      dyBorder;

extern HBITMAP hbmBitmaps;
extern HBITMAP hbmDefault;
extern HDC     hdcMem;

extern DWORD   cx;
extern DWORD   cy;
extern LPTSTR   pCurrentServer;

extern HFONT   hfontHelv;
extern HFONT   hfontHelvBold;

extern DWORD   SysColorHighlight;
extern DWORD   SysColorWindow;
extern DWORD   SysColorBtnFace;
extern DWORD   SysColorBtnText;
extern DWORD   SysColorBtnShadow;

extern int iBackground;
extern int iBackgroundSel;
extern int iButtonFace;
extern int iButtonShadow;

extern TCHAR szInternational[];
extern BOOL TwentyFourHourClock;

// #ifdef JAPAN   /* v-hirot  July.07.1993 for New Prefix */
extern   BOOL  TimePrefix;
// #endif

extern BOOL     bJapan;

extern BOOL MetricMeasurement;
extern TCHAR szTimeSep[TIMESEP_LENGTH];
extern TCHAR szAMPM[2][AMPM_LENGTH];
extern BOOL TimeFormatLeadingZero;
extern TCHAR szDecimalPoint[2];
extern DWORD DecimalDigits;
extern DWORD LeadingZero;

extern HCURSOR hcursorArrow;
extern HCURSOR hcursorSizeWE;
extern HCURSOR hcursorWait;

extern HICON hiconPrinter;
extern HICON hiconServer;
extern HICON hiconConnect;
extern HICON hiconShared;

extern TCHAR szPrintManHlp[];
extern WCHAR szLPrintManHlp[];

extern UINT WM_Help;

extern BOOL bToolBar;

extern DWORD LocalPermission;

extern BOOL NetworkInstalled;

extern PTCHAR szWindows;
extern PTCHAR szDevices;
extern PTCHAR szDevice;

extern BOOL ExpectingNotifyChangeKeyValue;

//==========================================================================
//              External Function Declaration and Prototypes
//==========================================================================
/* arrow.c */
short ArrowVScrollProc(short wScroll, short nCurrent, LPARROWVSCROLL lpAVS);
BOOL OddArrowWindow(HWND);
BOOL ValidateEntry( HWND hwnd, WORD CtlId, LPARROWVSCROLL lpAVS );

/*
 *  Extern functions in dialogs.c
 */
//  Dialog procedures
BOOL APIENTRY DocDetailsDlg (HWND hWnd, UINT  usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY PrtPropDlg (HWND hWnd, UINT  usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY InstallDriverDlg (HWND hWnd, UINT  usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY SelectDriverDlg(HWND hwnd, UINT  msg, WPARAM wparam, LPARAM lparam);
BOOL APIENTRY RemovePrinterDlg(HWND hwnd, UINT  msg, WPARAM wparam, LPARAM lparam);
BOOL APIENTRY SelectMonitorDlg(HWND hwnd, UINT  msg, WPARAM wparam, LPARAM lparam);
BOOL CALLBACK GreyText( HWND hwnd, LPARAM lParam );

//  Other (These should probably be in UTIL.C */
DWORD ConvertChartoTime (LPTSTR string);
void  ConvertSystemTimeToChar(SYSTEMTIME *pSystemTime, LPTSTR   String);
void  ConvertTimetoChar (DWORD Hour, DWORD Minutes, LPTSTR string);
void  ConvertMinutestoChar (DWORD Minutes, LPTSTR string);
BOOL CreateNewPrinter( HWND hWnd, PPRT_PROP_DLG_DATA pPrtPropDlgData );
LPTSTR GetString(int id);
LPTSTR GetStringParms(int id, ...);
#ifdef  LATER
BOOL APIENTRY ConnectDlg (HWND hWnd, WORD usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY DeleteDriverDlg (HWND hWnd, WORD usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY SetServerFocusDlg (HWND hWnd, WORD usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY PrinterDlg (HWND hWnd, WORD usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY MorePrinterDlg (HWND hWnd, WORD usMsg, WPARAM wParam, LONG lParam);
BOOL APIENTRY NewDriverDlg (HWND hWnd, WORD usMsg, WPARAM wParam, LONG lParam);

//  Other
void  SetPrinterProperties (HWND hWnd, PQUEUE pQueue);
void  SetSuggestedFields (HWND hWnd, PQUEUE pQueue);
void  SetupJobProperties (HWND hWnd, PQUEUE pQueue);
void  SetupPrintProc (HWND hWnd, PQUEUE pQueue);
#endif  //  LATER


/*
 *  Extern functions in forms.c
 */
BOOL APIENTRY FormsDlg( HWND hwnd, UINT  msg, WPARAM wparam, LPARAM lparam );


/*
 *  Extern functions in printer.c
 */
PQUEUE AllocQueue (LPTSTR pPrinterName);
BOOL   CheckError (HWND hWnd, DWORD Error);
DWORD  DeleteQJ (HWND hwnd, PQUEUE pQueue);
BOOL   DestroyPrinter(PQUEUE pQueue);
VOID   DestroyMDIWinInfo( PMDIWIN_INFO pInfo );
void   ErrorMsgBox (HWND hwnd, LPTSTR lpszMsg);
BOOL   GetJobs( PVOID pContext, PDWORD pFlags );
BOOL   FreeQueue (PQUEUE pQueue);
BOOL   InitQueueInfo (PQUEUE pQueue, PDWORD pError);
DWORD PauseResumeQJ( HANDLE hPrinter, LPPRINTER_INFO_2 pPrinter,
                     LPJOB_INFO_2 pJobs, DWORD JobSelected );
BOOL OpenPrinterForSpecifiedAccess(
             LPTSTR   pName,
             LPHANDLE pHandle,
             DWORD    AccessRequested,
    OPTIONAL PDWORD   pAccessGranted );

/*
 *  Extern functions in printman.c
 */
//  WndProcs
long FrameWndProc (HWND hWnd, UINT message, WPARAM wParam, LONG lParam);
long APIENTRY MDIListWndProc (HWND hWnd, UINT msg, WPARAM wParam, LONG lParam);
long APIENTRY MDIWndProc (HWND hWnd, UINT msg, WPARAM wParam, LONG lParam);

//  Other
HWND CreateQueueWindow(
   HWND     hWnd,
   PQUEUE   pQueue,
   DWORD    Status,
   DWORD    WindowType,
   DWORD    Flags
);
void DisplayInfo( HWND hWnd, HDC hDC, LPRECT pPaintRect, PMDIWIN_INFO pMDIWinInfo);
void DrawLine (HDC hDC, LPRECT pRect, LPTSTR pStr, BOOL bInv);
BOOL InitApplication (HANDLE hInstance, LPHANDLE lphAccel);
BOOL InitQueueChildWindows (HWND  hWnd);
void Refresh (HWND hwnd, PMDIWIN_INFO pInfo, DWORD RepaintOption);
void SetJobButtonText (PQUEUE pQueue, DWORD Status);
void SetPrinterButtonText (PQUEUE pQueue, DWORD Status);
VOID UpdateStatus(HWND hWnd);
BOOL DisplayStatusIcon( HDC hdc, PRECT prect, int Datatype, PBYTE pData, BOOL Highlight );
int Message(HWND hwnd, DWORD Type, int CaptionID, int TextID, ...);
int GetErrorStringID( DWORD ErrorID );
FARPROC LoadLibraryGetProcAddress(HWND hwnd, LPTSTR LibraryName, LPCSTR ProcName,
                                  PHANDLE phLibrary);
LPBYTE GetPrinterInfo(          IN  DWORD   Flags,
                                IN  LPTSTR  Name,
                                IN  DWORD   Level,
                                IN  LPBYTE  pPrinters,
                                OUT LPDWORD pcbPrinters,
                                OUT LPDWORD pcReturned,
                       OPTIONAL OUT LPDWORD pcbNeeded,
                       OPTIONAL OUT PDWORD  pError );
int GetJobStatusString( DWORD Status, LPTSTR string );

VOID
SetWindowTypeIcon(
    PQUEUE pQueue,
    DWORD WindowType,
    DWORD Flags);

VOID FrameCommandRefresh(HWND hWnd);


/*
 *  Extern functions in toolbar.c
 */
HWND PMCreateToolbar( BOOL NetworkAccess );
void CheckTBButton(DWORD idCommand);
void EnableCheckTBButtons(HWND hwndActive);
void GetDefaultLabelExtent(HWND hWnd);
BOOL LoadBitmaps();
void LoadDefaultLabel();
LONG ToolbarWndProc(HWND hWnd, UINT wMsg, DWORD wParam, LONG lParam);
BOOL AddToPrinterList (HWND  hwnd);
void UpdateDefaultList ();
VOID ToolbarCommandSelChange(BOOL bCompareDefault);

/*
 *  Extern functions in util.c
 */
LPVOID AllocSplMem (DWORD cb);
LPTSTR  AllocSplStr (LPTSTR lpStr);
BOOL   FreeSplMem (LPVOID pMem);
BOOL   FreeSplStr (LPTSTR lpStr);
LPVOID ReallocSplMem (LPVOID lpOldMem, DWORD cbNew);
BOOL   ReallocSplStr (LPTSTR *plpStr, LPTSTR lpStr);
BOOL EnumGeneric( IN  PROC    fnEnum,
                  IN  DWORD   Level,
                  IN  PBYTE   *ppEnumData,
                  IN  DWORD   cbBuf,
                  OUT LPDWORD pcbReturned,
                  OUT LPDWORD pcReturned,
                  IN  PVOID   Arg1,
                  IN  PVOID   Arg2,
                  IN  PVOID   Arg3 );

BOOL GetGeneric( IN  PROC    fnGet,
                 IN  DWORD   Level,
                 IN  PBYTE   *ppGetData,
                 IN  DWORD   cbBuf,
                 OUT LPDWORD pcbReturned,
                 IN  PVOID   Arg1,
                 IN  PVOID   Arg2 );

VOID ShowHelp(HWND hWnd, UINT Type, DWORD Data);


LONG  GetCountryCode(VOID);

LPTSTR
GetErrorString(
    DWORD   Error
);

DWORD ReportFailure( HWND  hwndParent,
                     DWORD idInsufficientPrivilege,
                     DWORD idDefaultError );

LPWSTR GetUnicodeString(int id);

LPTSTR Make8dot3Name( LPTSTR pName );
BOOL Is8dot3Name( LPTSTR pName );


/*
 *  Extern functions in registry.c
 */
DWORD WriteRegistryData( LPTSTR          pEntryNode,
                         LPTSTR          pEntryName,
                         LPBYTE          pData,
                         PREGISTRY_ENTRY pEntry );

DWORD ReadRegistryData( LPTSTR          pEntryNode,
                        LPTSTR          pEntryName,
                        LPBYTE          pData,
                        PREGISTRY_ENTRY pEntry );

typedef DWORD (CALLBACK *ENUMREGPROC)( PVOID pContext, PTCHAR pEnumData, PVOID pvParam );

DWORD EnumRegistryValues( LPTSTR      pEntryNode,
                          ENUMREGPROC pfnEnum,
                          PVOID       pContext );

DWORD DeleteRegistryValues( LPTSTR pNode );


/*
 *  Extern functions in security.c
 */
BOOL GetMaximumServerAccess( LPTSTR  pServerName,
                             PDWORD  pAccessGranted,
                             PHANDLE phServer OPTIONAL );
void CallDiscretionaryAclEditor( HWND hwnd, PQUEUE pPrinterContext );
void CallSystemAclEditor(HWND hwnd, PQUEUE pPrinterContext);
void CallTakeOwnershipDialog(HWND hwnd, PQUEUE pPrinterContext);
BOOL CheckPrinterAccessForAuditing( LPTSTR pPrinterName, PHANDLE phPrinter );

/*
 *  Extern functions in reconn.c
 */

DWORD RemoveFromReconnectList(LPTSTR pszRemotePath) ;
DWORD AddToReconnectList(LPTSTR pszRemotePath) ;


/*
 *  Extern functions in threads.c
 */


#if DBG

VOID DbgProtected( DWORD OwningThread );
VOID DbgNotProtected( DWORD OwningThread );

#endif /* DBG */


DWORD
OpenThreadObject(
    LPWSTR pszName,
    PHANDLE phPrinter,
    PDWORD pdwAccessGranted,
    DWORD WindowType );

VOID
ReopenPrinter(
    PVOID pContext,
    DWORD WindowType,
    BOOL bWait);

VOID
RefreshThread(
   PMDIWIN_INFO pInfo
);

VOID
LocalServerThread(
    HANDLE hServer
);

VOID
InitQueueThread(
    PVOID pContext
);

VOID
InitServerWindowThread(
    PVOID pContext
);

BOOL
CheckQuitQueue(
    PVOID pContext);

#if DBG

VOID EnterProtectedData( PMDIWIN_INFO pInfo );
VOID LeaveProtectedData( PMDIWIN_INFO pInfo );
VOID EnterProtectedHandle( PMDIWIN_INFO pInfo );
VOID LeaveProtectedHandle( PMDIWIN_INFO pInfo );

#define DBG_IN_PROTECTED_DATA( pInfo )  DbgProtected( pInfo->DataMutexOwner )
#define DBG_OUT_PROTECTED_DATA( pInfo ) DbgNotProtected( pInfo->DataMutexOwner )
#define DBG_IN_PROTECTED_HANDLE( pInfo )
#define DBG_OUT_PROTECTED_HANDLE( pInfo )

#define ENTER_PROTECTED_DATA( pInfo )   EnterProtectedData( pInfo )
#define LEAVE_PROTECTED_DATA( pInfo )   LeaveProtectedData( pInfo )
#define ENTER_PROTECTED_HANDLE( pInfo )
#define LEAVE_PROTECTED_HANDLE( pInfo )

#else /* NOT DBG */

#define DBG_IN_PROTECTED_DATA( pInfo )
#define DBG_OUT_PROTECTED_DATA( pInfo )
#define DBG_IN_PROTECTED_HANDLE( pInfo )
#define DBG_OUT_PROTECTED_HANDLE( pInfo )

#define ENTER_PROTECTED_DATA( pInfo )   WaitForSingleObject( pInfo->DataMutex, INFINITE )
#define LEAVE_PROTECTED_DATA( pInfo )   ReleaseMutex( pInfo->DataMutex )
#define ENTER_PROTECTED_HANDLE( pInfo )
#define LEAVE_PROTECTED_HANDLE( pInfo )

#endif /* DBG */

#define NULLC TEXT('\0')
#define SPACE TEXT(' ')
#define BACKSLASH TEXT('\\')
#define NEWLINE TEXT('\n')
#define RETURN TEXT('\r')
#define TAB TEXT('\t')

#define THOUSANDSEPSWITCH     32768
#define SPACE_CHAR            TEXT(' ')
#define COMMA_CHAR            TEXT(',')
extern  TCHAR ThousandSeparator;

ULONG FormatFileSize (IN DWORD rgfSwitchs, IN PLARGE_INTEGER FileSize, IN DWORD Width, OUT PTCHAR FormattedSize);
