/*++

Copyright (c) 1990  Microsoft Corporation

Module Name:

    scbind.c

Abstract:

    Contains the RPC bind and un-bind routines for the Service Controller.

Author:

    Dan Lafferty (danl)     19-Mar-1991

Environment:

    User Mode -Win32

Revision History:

    19-Mar-1991     danl
        created

--*/

//
// INCLUDES
//
#include <nt.h>         // DbgPrint prototype
#include <rpc.h>        // DataTypes and runtime APIs
#include <svcctl.h>     // generated by the MIDL complier 
#include <ntrpcp.h>     // RpcUtils for binding

#include <sclib.h>      // ansi to unicode conversion functions.
#include <scdebug.h>    // SCC_LOG
#include <sclib.h>      // ScConvertToUnicode

//
// GLOBALS      -   This handle maintains a static binding for status
//                  messages between the service process and the service 
//                  controller.
//
    static handle_t StatusBindHandle;


RPC_STATUS
InitializeStatusBinding( VOID)

/*++

Routine Description:

    This routine initializes the global StatusBindHandle that is used for
    the status connection to the service controller.  This routine is called
    by the Service Process (NetServiceStartCtrlDispatcher) when it starts up.
    

Arguments:

    none

Return Value:

    NERR_Success - (or 0) if the operation was successful.  

    Otherwise, it returns any RPC failure status that can be returned
    from RpcBindToInterface.


--*/
{
    RPC_STATUS      status;

    status = RpcpBindRpc (
                NULL,
                L"svcctl",
                0,  
                &StatusBindHandle);

    SCC_LOG(TRACE,"InitializeStatusBinding:RpcpBindRpc status=%d\n", status);

    SCC_LOG(TRACE,"InitializeStatusBinding: handle=%d\n",StatusBindHandle);

    return(status);
}


/****************************************************************************/
handle_t
RPC_SERVICE_STATUS_HANDLE_bind (
    SERVICE_STATUS_HANDLE   hService)

/*++

Routine Description:

    This bind routine is used for SetServiceStatus calls only.

    Currently all this routine does is return the global handle that was
    obtained when InitializeStatusBinding() was called.  It is desirable
    to have a single status binding that is done once during the life
    of the service process.

Arguments:
    
    ServerName - A pointer to a string containing the name of the server
        to bind with.

Return Value:

    The binding handle is returned to the stub routine.  If the 
    binding is unsuccessful, a NULL will be returned.

--*/
{

    UNREFERENCED_PARAMETER(hService);

    SCC_LOG(TRACE,"SERVICE_STATUS_HANDLE_bind: handle=%d\n",StatusBindHandle);
    return (StatusBindHandle);
}   



/****************************************************************************/
void
RPC_SERVICE_STATUS_HANDLE_unbind (
    SERVICE_STATUS_HANDLE   hService,
    handle_t                BindingHandle)

/*++

Routine Description:

    This un-bind routine is used for SetServiceStatus calls only.
    The RPC client stubs call this routine at the end of each RPC call
    in order to terminate the binding.

    It is desirable to maintain a static binding with the 
    service controller throughout the life of the service process.
    Therefore this binding is never terminated.  The termination of the 
    service process is expected to dispose of this binding.

Arguments:

    ServerName - This is the name of the server from which to unbind.    

    BindingHandle - This is the binding handle that is to be closed.

Return Value:
    
    none.

--*/
{
    SCC_LOG(TRACE,"SERVICE_STATUS_HANDLE_unbind: handle=%d\n",BindingHandle);
    SCC_LOG(TRACE,"We do not actually unbind this handle\n",0);

    UNREFERENCED_PARAMETER (BindingHandle);
    UNREFERENCED_PARAMETER (hService);

    return;

}


/****************************************************************************/
handle_t
SVCCTL_HANDLEW_bind (
    SVCCTL_HANDLEW   ServerName)

/*++

Routine Description:
    This routine calls a common bind routine that is shared by all services.
    This routine is called from the service controller client stubs when
    it is necessary to bind to a server.

Arguments:

    ServerName - A pointer to a string containing the name of the server
        to bind with.

Return Value:

    The binding handle is returned to the stub routine.  If the
    binding is unsuccessful, a NULL will be returned.

--*/
{
    handle_t    bindingHandle;
    RPC_STATUS  status;

    status = RpcpBindRpc (
                ServerName,
                L"svcctl",
                L"Security=Impersonation Dynamic False",
                &bindingHandle);

    SCC_LOG(TRACE,"SVCCTL_HANDLEW_bind:RpcpBindRpc status=%d\n",status);
    SCC_LOG(TRACE,"SVCCTL_HANDLEW_bind: handle=%d\n",bindingHandle);

    return( bindingHandle);
}



/****************************************************************************/
void
SVCCTL_HANDLEW_unbind (
    SVCCTL_HANDLEW   ServerName,
    handle_t         BindingHandle)

/*++

Routine Description:

    This routine calls a common unbind routine that is shared by
    all services.
    This routine is called from the Service Controller client stubs when
    it is necessary to unbind to a server.


Arguments:

    ServerName - This is the name of the server from which to unbind.

    BindingHandle - This is the binding handle that is to be closed.

Return Value:

    none.

--*/
{
    UNREFERENCED_PARAMETER(ServerName);     // This parameter is not used

    SCC_LOG(TRACE,"SVCCTL_HANDLEW_unbind: handle=%d\n",BindingHandle);

    RpcpUnbindRpc ( BindingHandle);
    return;
}

/****************************************************************************/
handle_t
SVCCTL_HANDLEA_bind (
    SVCCTL_HANDLEA   ServerName)

/*++

Routine Description:
    This routine calls a common bind routine that is shared by all services.
    This routine is called from the service controller client stubs when
    it is necessary to bind to a server.

Arguments:

    ServerName - A pointer to a string containing the name of the server
        to bind with.

Return Value:

    The binding handle is returned to the stub routine.  If the
    binding is unsuccessful, a NULL will be returned.

--*/
{
    handle_t    bindingHandle;
    RPC_STATUS  status;
    LPWSTR      pServerNameW = NULL;

    if (ServerName != NULL) {
        if (!ScConvertToUnicode(&pServerNameW, ServerName)) {
            SCC_LOG(ERROR,"SVCCTL_HANDLEA_bind:ScConvertToUnicode failed\n",0);
            return(NULL);
        }
    }
    
    status = RpcpBindRpc (
                pServerNameW,
                L"svcctl",
                L"Security=Impersonation Dynamic False",
                &bindingHandle);

    (void) LocalFree(pServerNameW);

    SCC_LOG(TRACE,"SVCCTL_HANDLEA_bind:RpcpBindRpc status=%d\n",status);
    SCC_LOG(TRACE,"SVCCTL_HANDLEA_bind: handle=%d\n",bindingHandle);

    return( bindingHandle);
}



/****************************************************************************/
void
SVCCTL_HANDLEA_unbind (
    SVCCTL_HANDLEA   ServerName,
    handle_t         BindingHandle)

/*++

Routine Description:

    This routine calls a common unbind routine that is shared by
    all services.
    This routine is called from the Service Controller client stubs when
    it is necessary to unbind to a server.


Arguments:

    ServerName - This is the name of the server from which to unbind.

    BindingHandle - This is the binding handle that is to be closed.

Return Value:

    none.

--*/
{
    UNREFERENCED_PARAMETER(ServerName);     // This parameter is not used

    SCC_LOG(TRACE,"SVCCTL_HANDLEA_unbind: handle=%d\n",BindingHandle);

    RpcpUnbindRpc ( BindingHandle);
    return;
}



