//+------------------------------------------------------------------------
//
//  File:       strtype.c
//
//  Contents:   widechar character type function (CT_CTYPE1) and (CT_CTYPE3)
//
//  Synopsis:   We do not have wide char support for IsChar functions
//              under Win95.  The Unicode-Wrapper functions we have
//              in core\wrappers all convert to CP_ACP and then call
//              the A version, which means we will have invalid results
//              for any characters which aren't in CP_ACP.
//
//              The solution is to roll our own, which result in these
//              unfortunately large tables.  Here's how it works:
//
//              bits:   fedc ba98 7654 3210
//                      pppp pppp iiib bbbb
//
//              The 'b' bits form a 32-bit bit mask into our data.  The data
//              entrys boolean, and are thus 4-bytes long.  Of the 2^32
//              possible combinations, we in fact have only 218 distinct
//              values of data.  These are stored in adwData.
//
//              The 'p' bits represent a page.  Each page has eight
//              possible entries, represent by 'i'.  In most pages, the
//              bitfields and data are both uniform.
//
//              adwData[abIndex[abType[page]][index]] represents the data
//
//              1 << bits represents the bitmask.
//
//-------------------------------------------------------------------------
#include "priv.h"

#define __BIT_SHIFT 0
#define __INDEX_SHIFT 5
#define __PAGE_SHIFT 8

#define __BIT_MASK 31
#define __INDEX_MASK 7

// straight lookup functions are inlined.

#define ISCHARFUNC(type, wch) \
    (adwData[abIndex[abType1##type[wch>>__PAGE_SHIFT]] \
                          [(wch>>__INDEX_SHIFT)&__INDEX_MASK]] \
            >> (wch&__BIT_MASK)) & 1 
    
const DWORD adwData[218] =
{
#if 0
    0x00000000, 0x7fffffe0, 0xfffffeff, 0xffffffff,  // 0x00-0x03
    0xfffffd3f, 0xffffff00, 0x0000ffff, 0xff800000,  // 0x04-0x07
    0x02ebffff, 0xdfffffff, 0xfffefe2a, 0xbffff000,  // 0x08-0x0b
    0x7ffbffff, 0xffff7ffb, 0xd000ffff, 0xf998ffff,  // 0x0c-0x0f
    0xfff3fdd0, 0x00007fff, 0xfffffe00, 0x7fffffff,  // 0x10-0x13
    0xff000000, 0xffe0e000, 0x7fe00000, 0xffffff3e,  // 0x14-0x17
    0xfffef400, 0x07ffffff, 0xffffffd0, 0x000000ff,  // 0x18-0x1b
    0xd0000000, 0x07f99fff, 0xffbfa3d0, 0x0000000d,  // 0x1c-0x1f
    0xd000d000, 0x07e19fff, 0xffbfb6d0, 0x0000007a,  // 0x20-0x23
    0x07f5dfff, 0xffbfb7d0, 0x80000000, 0xffbfb3d0,  // 0x24-0x27
    0x07e3bd6b, 0x18e3fdd0, 0x07fbbfff, 0xffbff7d0,  // 0x28-0x2b
    0x00000002, 0xffbfffd0, 0xfffeb000, 0xfd000000,  // 0x2c-0x2f
    0x69a40f7f, 0x75360000, 0x0000000d, 0xfd00ffff,  // 0x30-0x33
    0xffffffd1, 0xe0ffffff, 0xffffffe0, 0xfffffdfd,  // 0x34-0x37
    0xfdfdff55, 0xfffffffd, 0xfffffbf8, 0x3bf8f3f0,  // 0x38-0x3b
    0xfff83bf8, 0x0000000f, 0xffd00000, 0x04000000,  // 0x3c-0x3f
    0xfffff81e, 0xffffffee, 0xfff87fff, 0xfffe0000,  // 0x40-0x43
    0xf0000000, 0xfffd0000, 0xfe001f01, 0xfffffefa,  // 0x44-0x47
    0xdbffffff, 0xffffd000, 0x00001fff, 0xffbfffff,  // 0x48-0x4b
    0xbfffffff, 0xffff3fff, 0x0000fff0, 0xfffffff8,  // 0x4c-0x4f
    0x03ffffff, 0x7ffffffe, 0x3f3f3f38, 0x007d0000,  // 0x50-0x53
    0xfff00000, 0x00000001, 0x7fff003f, 0x8000001f,  // 0x54-0x57
    0x8000001e, 0x00000100, 0x00002000, 0xfffffffe,  // 0x58-0x5b
    0xd0000d22, 0x0d000000, 0x3e000000, 0x0000007f,  // 0x5c-0x5f
    0x00400000, 0x00001800, 0x00080011, 0x801fe000,  // 0x60-0x63
    0x003d8000, 0x00000bff, 0x70000000, 0xfffdf800,  // 0x64-0x67
    0x3d008000, 0x0000000b, 0xf99d0100, 0x30003fe0,  // 0x68-0x6b
    0x20000000, 0xe19d0000, 0x0000f800, 0xfddd8000,  // 0x6c-0x6f
    0xf19d0300, 0x00008000, 0x30000000, 0x00000003,  // 0x70-0x73
    0xe3bd0100, 0xfbbd0600, 0xf3bd0100, 0x00014fe1,  // 0x74-0x77
    0x03ff0030, 0x0001ffdd, 0xfafd0000, 0x00000010,  // 0x78-0x7b
    0x00000007, 0xd0070007, 0x00070006, 0xfffeffff,  // 0x7c-0x7f
    0xfe000000, 0x003f003f, 0x003e0000, 0xffffff80,  // 0x80-0x83
    0xe000ffff, 0xffe00000, 0xf8000000, 0xfffffd00,  // 0x84-0x87
    0xfffbffff, 0xffff0000, 0xfffff03f, 0x7bdfffff,  // 0x88-0x8b
    0xff7fffff, 0xfff5e2fe, 0x7f0003ff, 0xfffff8ff,  // 0x8c-0x8f
    0xffff7ffe, 0xffffff01, 0x0000007e, 0x0000001e,  // 0x90-0x93
    0xfffffff1, 0xffff8000, 0xfff0ffff, 0xfffffe1f,  // 0x94-0x97
    0xf000ffff, 0xf87fefff, 0xfef0ebff, 0x7d008000,  // 0x98-0x9b
    0xfefe0000, 0x0000ffd0, 0x00003040, 0x03ff0000,  // 0x9c-0x9f
    0x01ffe000, 0x00008fd0, 0x7e000000, 0xfffffefe,  // 0xa0-0xa3
    0xaaaaaaaa, 0xaaaaaa55, 0x552aaaaa, 0xaaaaaad4,  // 0xa4-0xa7
    0x6b73db8d, 0xa94b7588, 0x0db55552, 0xaaaa682a,  // 0xa8-0xab
    0xaaaaaa00, 0x02eb7fff, 0xdff00000, 0x0000382a,  // 0xac-0xaf
    0xaaaa0000, 0x8000aaaa, 0x5110aaaa, 0xaaa2a880,  // 0xb0-0xb3
    0xaaaaa800, 0xaaaaaa80, 0x00ff00fd, 0x00ff00ff,  // 0xb4-0xb7
    0x00fd0055, 0x00ff0000, 0x00ff00f8, 0x00f800f0,  // 0xb8-0xbb
    0x00f800f8, 0x000003ff, 0x55555555, 0x555555aa,  // 0xbc-0xbf
    0xaad55555, 0x5555552b, 0x948d2472, 0x54948a64,  // 0xc0-0xc3
    0x06daaaad, 0x5555b415, 0x55555500, 0xdfdff5ff,  // 0xc4-0xc7
    0x3ffef026, 0x9f800000, 0x000fffff, 0xfffed600,  // 0xc8-0xcb
    0x1555f000, 0x40005555, 0x28885555, 0x55515440,  // 0xcc-0xcf
    0x555557e0, 0x55555540, 0xff00fd00, 0xff00ff00,  // 0xd0-0xd3
    0xfd00ff00, 0xff00fffd, 0xff00fb00, 0x3b00f300,  // 0xd4-0xd7
    0xff003b00, 0xfe001f00
#else
    0x00000000, 0x07fffffe, 0xff7fffff, 0xffffffff,  // 0x00-0x03
    0xfc3fffff, 0x00ffffff, 0xffff0000, 0x000001ff,  // 0x04-0x07
    0xffffd740, 0xfffffffb, 0x547f7fff, 0x000ffffd,  // 0x08-0x0b
    0xffffdffe, 0xdffeffff, 0xffff0003, 0xffff199f,  // 0x0c-0x0f
    0x033fcfff, 0xfffe0000, 0x007fffff, 0xfffffffe,  // 0x10-0x13
    0x000000ff, 0x000707ff, 0x000007fe, 0x7cffffff,  // 0x14-0x17
    0x002f7fff, 0xffffffe0, 0x03ffffff, 0xff000000,  // 0x18-0x1b
    0x00000003, 0xfff99fe0, 0x03c5fdff, 0xb0000000,  // 0x1c-0x1f
    0x00030003, 0xfff987e0, 0x036dfdff, 0x5e000000,  // 0x20-0x23
    0xfffbafe0, 0x03edfdff, 0x00000001, 0x03cdfdff,  // 0x24-0x27
    0xd63dc7e0, 0x03bfc718, 0xfffddfe0, 0x03effdff,  // 0x28-0x2b
    0x40000000, 0x03fffdff, 0x000d7fff, 0x0000003f,  // 0x2c-0x2f
    0xfef02596, 0x00006cae, 0x30000000, 0xffff003f,  // 0x30-0x33
    0x83ffffff, 0xffffff07, 0x07ffffff, 0x3f3fffff,  // 0x34-0x37
    0xaaff3f3f, 0x3fffffff, 0x1fdfffff, 0x0fcf1fdc,  // 0x38-0x3b
    0x1fdc1fff, 0xf0000000, 0x000003ff, 0x00000020,  // 0x3c-0x3f
    0x781fffff, 0x77ffffff, 0xfffe1fff, 0x00007fff,  // 0x40-0x43
    0x0000000f, 0x00003fff, 0x80f8007f, 0x5f7fffff,  // 0x44-0x47
    0xffffffdb, 0x0003ffff, 0xfff80000, 0xfffffdff,  // 0x48-0x4b
    0xfffffffd, 0xfffcffff, 0x0fff0000, 0x1fffffff,  // 0x4c-0x4f
    0xffffffc0, 0x7ffffffe, 0x1cfcfcfc, 0x00003e00,  // 0x50-0x53
    0x00000fff, 0x80000000, 0xfc00fffe, 0xf8000001,  // 0x54-0x57
    0x78000001, 0x00800000, 0x00040000, 0x7fffffff,  // 0x58-0x5b
    0x44300003, 0x000000b0, 0x0000007c, 0xfe000000,  // 0x5c-0x5f
    0x00000200, 0x00180000, 0x88001000, 0x0007f801,  // 0x60-0x63
    0x00013c00, 0xffd00000, 0x0000000e, 0x001f3fff,  // 0x64-0x67
    0x0001003c, 0xd0000000, 0x0080399f, 0x07fc000c,  // 0x68-0x6b
    0x00000004, 0x00003987, 0x001f0000, 0x00013bbf,  // 0x6c-0x6f
    0x00c0398f, 0x00010000, 0x0000000c, 0xc0000000,  // 0x70-0x73
    0x00803dc7, 0x00603ddf, 0x00803dcf, 0x87f28000,  // 0x74-0x77
    0x0c00ffc0, 0x3bff8000, 0x00003f5f, 0x08000000,  // 0x78-0x7b
    0xe0000000, 0xe000e003, 0x6000e000, 0xffff7fff,  // 0x7c-0x7f
    0x0000007f, 0xfc00fc00, 0x00007c00, 0x01ffffff,  // 0x80-0x83
    0xffff0007, 0x000007ff, 0x0000001f, 0x003fffff,  // 0x84-0x87
    0xffffdfff, 0x0000ffff, 0xfc0fffff, 0xfffff3de,  // 0x88-0x8b
    0xfffffeff, 0x7f47afff, 0xffc000fe, 0xff1fffff,  // 0x8c-0x8f
    0x7ffeffff, 0x80ffffff, 0x7e000000, 0x78000000,  // 0x90-0x93
    0x8fffffff, 0x0001ffff, 0xffff0fff, 0xf87fffff,  // 0x94-0x97
    0xffff000f, 0xfff7fe1f, 0xffd70f7f, 0x0001003e,  // 0x98-0x9b
    0x00007f7f, 0x03ff0000, 0x020c0000, 0x0000ffc0,  // 0x9c-0x9f
    0x0007ff80, 0x03f10000, 0x0000007e, 0x7f7fffff,  // 0xa0-0xa3
    0x55555555, 0xaa555555, 0x555554aa, 0x2b555555,  // 0xa4-0xa7
    0xb1dbced6, 0x11aed295, 0x4aaaadb0, 0x54165555,  // 0xa8-0xab
    0x00555555, 0xfffed740, 0x00000ffb, 0x541c0000,  // 0xac-0xaf
    0x00005555, 0x55550001, 0x5555088a, 0x01154555,  // 0xb0-0xb3
    0x00155555, 0x01555555, 0x3f00ff00, 0xff00ff00,  // 0xb4-0xb7
    0xaa003f00, 0x0000ff00, 0x1f00ff00, 0x0f001f00,  // 0xb8-0xbb
    0x1f001f00, 0xffc00000, 0xaaaaaaaa, 0x55aaaaaa,  // 0xbc-0xbf
    0xaaaaab55, 0xd4aaaaaa, 0x4e243129, 0x2651292a,  // 0xc0-0xc3
    0xb5555b60, 0xa82daaaa, 0x00aaaaaa, 0xffaffbfb,  // 0xc4-0xc7
    0x640f7ffc, 0x000001f9, 0xfffff000, 0x00637fff,  // 0xc8-0xcb
    0x000faaa8, 0xaaaa0002, 0xaaaa1114, 0x022a8aaa,  // 0xcc-0xcf
    0x07eaaaaa, 0x02aaaaaa, 0x003f00ff, 0x00ff00ff,  // 0xd0-0xd3
    0x00ff003f, 0x3fff00ff, 0x00df00ff, 0x00cf00dc,  // 0xd4-0xd7
    0x00dc00ff, 0x00f8007f
#endif
};

const BYTE abIndex[98][8] =
{
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x00
    { 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x02, 0x02 }, // 0x01
    { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x04 }, // 0x02
    { 0x05, 0x00, 0x06, 0x03, 0x03, 0x07, 0x00, 0x00 }, // 0x03
    { 0x00, 0x00, 0x00, 0x00, 0x08, 0x09, 0x0a, 0x0b }, // 0x04
    { 0x0c, 0x03, 0x0d, 0x03, 0x0e, 0x03, 0x0f, 0x10 }, // 0x05
    { 0x00, 0x11, 0x12, 0x13, 0x14, 0x00, 0x06, 0x15 }, // 0x06
    { 0x00, 0x01, 0x16, 0x11, 0x03, 0x17, 0x18, 0x00 }, // 0x07
    { 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20 }, // 0x08
    { 0x21, 0x22, 0x23, 0x00, 0x24, 0x25, 0x00, 0x26 }, // 0x09
    { 0x1d, 0x27, 0x1f, 0x1c, 0x28, 0x29, 0x00, 0x00 }, // 0x0a
    { 0x2a, 0x2b, 0x00, 0x1c, 0x2a, 0x2b, 0x2c, 0x1c }, // 0x0b
    { 0x2a, 0x2d, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00 }, // 0x0c
    { 0x13, 0x2e, 0x2f, 0x00, 0x30, 0x31, 0x32, 0x00 }, // 0x0d
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x33, 0x12 }, // 0x0e
    { 0x03, 0x03, 0x34, 0x03, 0x03, 0x35, 0x03, 0x1a }, // 0x0f
    { 0x03, 0x03, 0x03, 0x03, 0x36, 0x03, 0x03, 0x1a }, // 0x10
    { 0x37, 0x03, 0x38, 0x39, 0x03, 0x3a, 0x3b, 0x3c }, // 0x11
    { 0x00, 0x00, 0x00, 0x00, 0x3d, 0x03, 0x03, 0x3e }, // 0x12
    { 0x3f, 0x00, 0x13, 0x03, 0x40, 0x13, 0x03, 0x41 }, // 0x13
    { 0x19, 0x42, 0x03, 0x03, 0x43, 0x00, 0x00, 0x00 }, // 0x14
    { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03 }, // 0x15
    { 0x03, 0x03, 0x03, 0x03, 0x03, 0x2f, 0x00, 0x00 }, // 0x16
    { 0x03, 0x03, 0x03, 0x03, 0x03, 0x44, 0x00, 0x00 }, // 0x17
    { 0x03, 0x45, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x18
    { 0x46, 0x47, 0x48, 0x03, 0x03, 0x49, 0x4a, 0x4b }, // 0x19
    { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x4c }, // 0x1a
    { 0x03, 0x39, 0x06, 0x03, 0x4d, 0x03, 0x14, 0x4e }, // 0x1b
    { 0x00, 0x00, 0x00, 0x00, 0x03, 0x03, 0x03, 0x4f }, // 0x1c
    { 0x00, 0x01, 0x01, 0x50, 0x03, 0x51, 0x52, 0x00 }, // 0x1d
    { 0x53, 0x26, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00 }, // 0x1e
    { 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x1f
    { 0x26, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x20
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55 }, // 0x21
    { 0x00, 0x56, 0x57, 0x58, 0x00, 0x13, 0x59, 0x59 }, // 0x22
    { 0x00, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00 }, // 0x23
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x5b, 0x3e }, // 0x24
    { 0x03, 0x03, 0x2f, 0x5c, 0x5d, 0x00, 0x00, 0x00 }, // 0x25
    { 0x00, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00 }, // 0x26
    { 0x00, 0x00, 0x5f, 0x00, 0x60, 0x06, 0x44, 0x61 }, // 0x27
    { 0x62, 0x00, 0x63, 0x64, 0x00, 0x00, 0x65, 0x45 }, // 0x28
    { 0x66, 0x3d, 0x67, 0x68, 0x66, 0x69, 0x6a, 0x6b }, // 0x29
    { 0x6c, 0x69, 0x6d, 0x6e, 0x66, 0x3d, 0x6f, 0x00 }, // 0x2a
    { 0x66, 0x3d, 0x70, 0x71, 0x72, 0x73, 0x74, 0x00 }, // 0x2b
    { 0x66, 0x73, 0x75, 0x00, 0x72, 0x73, 0x75, 0x00 }, // 0x2c
    { 0x72, 0x73, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x2d
    { 0x00, 0x77, 0x78, 0x00, 0x00, 0x79, 0x7a, 0x00 }, // 0x2e
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7b }, // 0x2f
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x7d, 0x7e }, // 0x30
    { 0x03, 0x7f, 0x80, 0x81, 0x82, 0x54, 0x06, 0x1c }, // 0x31
    { 0x03, 0x83, 0x4a, 0x03, 0x84, 0x03, 0x03, 0x85 }, // 0x32
    { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x49 }, // 0x33
    { 0x4c, 0x03, 0x03, 0x36, 0x00, 0x00, 0x00, 0x00 }, // 0x34
    { 0x03, 0x86, 0x85, 0x03, 0x03, 0x03, 0x03, 0x85 }, // 0x35
    { 0x03, 0x03, 0x03, 0x03, 0x87, 0x88, 0x03, 0x89 }, // 0x36
    { 0x8a, 0x03, 0x03, 0x89, 0x00, 0x00, 0x00, 0x00 }, // 0x37
    { 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x00, 0x00 }, // 0x38
    { 0x13, 0x91, 0x00, 0x00, 0x92, 0x00, 0x00, 0x93 }, // 0x39
    { 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00 }, // 0x3a
    { 0x4f, 0x03, 0x44, 0x94, 0x03, 0x95, 0x96, 0x5b }, // 0x3b
    { 0x03, 0x03, 0x03, 0x97, 0x03, 0x03, 0x39, 0x5b }, // 0x3c
    { 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x3d
    { 0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x3e
    { 0x00, 0x98, 0x99, 0x9a, 0x03, 0x03, 0x03, 0x4f }, // 0x3f
    { 0x56, 0x57, 0x58, 0x9b, 0x73, 0x26, 0x00, 0x9c }, // 0x40
    { 0x00, 0x9d, 0x00, 0x00, 0x00, 0x9e, 0x00, 0x00 }, // 0x41
    { 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x9d }, // 0x42
    { 0x00, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0x9f }, // 0x43
    { 0x00, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0xa0 }, // 0x44
    { 0x00, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0x00 }, // 0x45
    { 0x00, 0x00, 0x9d, 0x00, 0x00, 0x00, 0x9d, 0x00 }, // 0x46
    { 0x00, 0x00, 0x00, 0xa1, 0x3e, 0x00, 0x00, 0x00 }, // 0x47
    { 0x9d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x48
    { 0x00, 0x9d, 0xa2, 0xa2, 0x00, 0x00, 0x00, 0x00 }, // 0x49
    { 0x9d, 0xa2, 0xa2, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x4a
    { 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xa3, 0x00 }, // 0x4b
    { 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab }, // 0x4c
    { 0xac, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x4d
    { 0x00, 0x00, 0x00, 0x00, 0xad, 0xae, 0xaf, 0xb0 }, // 0x4e
    { 0x0c, 0x89, 0x00, 0xa4, 0xb1, 0xa4, 0xb2, 0xb3 }, // 0x4f
    { 0x00, 0x11, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x50
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x2f, 0x00 }, // 0x51
    { 0xa4, 0xa4, 0xa4, 0xa4, 0xb4, 0xa4, 0xa4, 0xb5 }, // 0x52
    { 0xb6, 0xb7, 0xb8, 0xb9, 0xb7, 0xba, 0xbb, 0xbc }, // 0x53
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0xbd, 0x89, 0x00 }, // 0x54
    { 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x55
    { 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x55, 0x02 }, // 0x56
    { 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5 }, // 0x57
    { 0xc6, 0x00, 0x06, 0xc7, 0xc8, 0xc9, 0x00, 0x00 }, // 0x58
    { 0x00, 0x00, 0x00, 0x00, 0x71, 0xca, 0xcb, 0xcc }, // 0x59
    { 0x00, 0x06, 0x0d, 0xbe, 0xcd, 0xbe, 0xce, 0xcf }, // 0x5a
    { 0x00, 0x00, 0x00, 0x13, 0x14, 0x00, 0x00, 0x00 }, // 0x5b
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x12 }, // 0x5c
    { 0xbe, 0xbe, 0xbe, 0xbe, 0xd0, 0xbe, 0xbe, 0xd1 }, // 0x5d
    { 0xd2, 0xd3, 0xd4, 0xd5, 0xd3, 0xd6, 0xd7, 0xd8 }, // 0x5e
    { 0x00, 0x00, 0x00, 0x00, 0x3d, 0x87, 0x06, 0x3e }, // 0x5f
    { 0xd9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // 0x60
    { 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00 }  // 0x61
};

const BYTE abType1Alpha[256] = // 154
{
    0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00,
    0x00, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x00,
    0x0e, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x11,
    0x00, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x13, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x16,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
    0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x17,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x15, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d
};

BOOL IsCharSpaceW(WCHAR wch)
{
    int nType;

    switch(wch>>8)
    {
        case 0x00: nType = 0x1e; break;
        case 0x20: nType = 0x1f; break;
        case 0x30: nType = 0x20; break;
        case 0xfe: nType = 0x21; break;
        default:   nType = 0x00; break;
    }

#if 0   
    return adwData[abIndex[nType][(wch>>__INDEX_SHIFT)&__INDEX_MASK]]
           & (1<<(wch&__BIT_MASK));
#endif
   
    return (adwData[abIndex[nType][(wch>>__INDEX_SHIFT)&__INDEX_MASK]]
            >>(wch&__BIT_MASK)) & 1;
}

const BYTE abType1Punct[256] = // 32
{
    0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x00,
    0x00, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x00,
    0x2f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30,
    0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x39, 0x3a, 0x3b, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3d, 0x00, 0x3e, 0x3f, 0x40
};

const BYTE abType1Digit[256] = // 11
{
    0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00,
    0x00, 0x43, 0x43, 0x44, 0x43, 0x45, 0x46, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48
};

BOOL IsCharDigitW(WCHAR wch) { return ISCHARFUNC(Digit, wch); }

BOOL IsCharXDigitW(WCHAR wch)
{
    int nType;

    switch(wch>>8)
    {
        case 0x00: nType = 0x49; break;
        case 0xff: nType = 0x4a; break;
        default:   nType = 0x00; break;
    }

    return (adwData[abIndex[nType][(wch>>__INDEX_SHIFT)&__INDEX_MASK]]
            >> (wch&__BIT_MASK)) & 1;
#if 0
    return adwData[abIndex[nType][(wch>>__INDEX_SHIFT)&__INDEX_MASK]]
           & (1<<(wch&__BIT_MASK));
#endif
}

const BYTE abType1Upper[256] = // 12
{
    0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x51, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x52, 0x53,
    0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55
};

const BYTE abType1Lower[256] = // 13
{
    0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x5c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5d, 0x5e,
    0x00, 0x00, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x61
};


BOOL IsCharPunctW(WCHAR wch) { return ISCHARFUNC(Punct, wch); }


BOOL IsCharCntrlW(WCHAR wch)
{
    return    (unsigned)(wch - 0x0000) <= (0x001f - 0x0000)
           || (unsigned)(wch - 0x007f) <= (0x009f - 0x007f);
}

// NB (cthrash) WCH_NBSP is considered blank, for compatibility.

BOOL IsCharBlankW(WCHAR wch)
{
    return    wch == 0x0009
           || wch == 0x0020
           || wch == 0x00a0
           || wch == 0x3000
           || wch == 0xfeff;
}


BOOL IsCharAlphaWrapW(WCHAR wch) { return ISCHARFUNC(Alpha, wch); }
BOOL IsCharUpperWrapW(WCHAR wch) { return ISCHARFUNC(Upper, wch); }
BOOL IsCharLowerWrapW(WCHAR wch) { return ISCHARFUNC(Lower, wch); }

BOOL IsCharAlphaNumericWrapW(WCHAR wch)
{
    return ISCHARFUNC(Alpha, wch) || ISCHARFUNC(Digit, wch);
}


static const BYTE abType3PageSub[256] = 
{
    0x00, 0x80, 0x81, 0x82, 0x00, 0x83, 0x84, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x85, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 
    0x00, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x8e, 0x8f, 0x90, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x91, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x20, 0x92, 0x00, 0x00, 0x93, 0x94, 0x00
};

static const BYTE abType3Page0[256] = 
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x08, 
    0x09, 0x09, 0x09, 0x09, 0x09, 0x08, 0x09, 0x09, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 
    0x09, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x09, 
    0x09, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x01, 0x01, 0x09, 0x09, 0x01, 0x09, 0x09, 0x01, 
    0x01, 0x01, 0x00, 0x01, 0x09, 0x01, 0x01, 0x09, 
    0x01, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const BYTE abType3Page32[256] = 
{
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 
    0x11, 0x11, 0x01, 0x01, 0x11, 0x11, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x09, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const BYTE abType3Page48[256] = 
{
    0x11, 0x11, 0x11, 0x00, 0x00, 0x20, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x11, 0x11, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01, 
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x06, 0x06, 0x06, 0x06, 0x06, 0x01, 0x01, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 
    0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 
    0x00, 0x06, 0x06, 0x16, 0x16, 0x04, 0x04, 0x00, 
    0x00, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 
    0x12, 0x12, 0x12, 0x12, 0x02, 0x12, 0x02, 0x12, 
    0x02, 0x12, 0x02, 0x12, 0x02, 0x12, 0x02, 0x12, 
    0x02, 0x12, 0x02, 0x12, 0x02, 0x12, 0x02, 0x12, 
    0x02, 0x12, 0x02, 0x12, 0x12, 0x02, 0x12, 0x02, 
    0x12, 0x02, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 
    0x02, 0x02, 0x12, 0x02, 0x02, 0x12, 0x02, 0x02, 
    0x12, 0x02, 0x02, 0x12, 0x02, 0x02, 0x12, 0x12, 
    0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 
    0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x02, 0x12, 
    0x02, 0x02, 0x12, 0x12, 0x02, 0x02, 0x02, 0x02, 
    0x02, 0x02, 0x02, 0x13, 0x06, 0x02, 0x02, 0x00
};

static const BYTE abType3Page255[256] = 
{
    0x00, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x10, 
    0x11, 0x11, 0x11, 0x11, 0x11, 0x10, 0x11, 0x11, 
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 
    0x11, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11, 0x11, 
    0x11, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11, 0x00, 
    0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0a, 0x0a, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0e, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 
    0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 
    0x00, 0x00, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00, 
    0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x00, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


struct tagType3DualValue
{
    DWORD   adwBitfield[8];
    DWORD   adwValue[2];
}
const aType3DualValue[21] =
{
    { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   // Page1
      0x00000000, 0x0000000f, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   // Page2
      0x00000000, 0x3f000000, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0x00000000, 0x00000000, 0x04000000, 0x000000b0,   // Page3
      0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0x00000000, 0xf8000000, 0x00000000, 0x00000200,   // Page5
      0x40000000, 0x00000009, 0x00180000, 0x00000000, 0x00000001 },
    { 0x88001000, 0x00000000, 0x00000000, 0x00003c00, 0x00000000,   // Page6
      0x00000000, 0x00100000, 0x00000200, 0x00000000, 0x00000001 },
    { 0x00000000, 0x80008000, 0x0c008040, 0x00000000, 0x00000000,   // Page14
      0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   // Page31
      0xe0000000, 0xe000e003, 0x6000e000, 0x00000000, 0x00000001 },
    { 0x00800000, 0x00000000, 0x00000000, 0x00000000, 0xffff0000,   // Page33
      0xffffffff, 0xffffffff, 0x000007ff, 0x00000000, 0x00000001 },
    { 0x40000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   // Page34
      0x00000000, 0x00000000, 0xfffc0000, 0x00000001, 0x00000000 },
    { 0x00000002, 0x00000000, 0x00000000, 0xf8000000, 0xffffffff,   // Page35
      0xffffffff, 0xffffffff, 0xffffffff, 0x00000001, 0x00000000 },
    { 0x00000000, 0xffffffe0, 0xfffff800, 0xffffffff, 0xffffffff,   // Page36
      0xffffffff, 0xffffffff, 0xffffffff, 0x00000001, 0x00000000 },
    { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffc00000,   // Page37
      0x00002000, 0x00000000, 0xffff8000, 0x00000001, 0x00000000 },
    { 0x03f00000, 0x00000000, 0x00000000, 0xffff0000, 0xffffffff,   // Page38
      0xffffffff, 0xffffffff, 0xffffffff, 0x00000001, 0x00000000 },
    { 0xfffff3de, 0xfffffeff, 0x7f47afff, 0x000000fe, 0xff100000,   // Page39
      0x7ffeffff, 0x00000000, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0xfffe0000, 0xffffffff, 0x0000001f, 0x00000000,   // Page49
      0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000010 },
    { 0x00000000, 0x00000000, 0x00000000, 0x80000000, 0x00000000,   // Page50
      0x00000000, 0x00000fff, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0x00000000, 0xff000000, 0x0001ffff, 0x00000000,   // Page51
      0x00000000, 0x00000000, 0x7fffffff, 0x00000000, 0x00000001 },
    { 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   // Page159
      0xffffffc0, 0xffffffff, 0xffffffff, 0x00000020, 0x00000000 },
    { 0x00000000, 0xffffc000, 0xffffffff, 0xffffffff, 0xffffffff,   // Page250
      0xffffffff, 0xffffffff, 0xffffffff, 0x00000020, 0x00000000 },
    { 0x00000000, 0xc0000000, 0x00000000, 0x00000000, 0x00000000,   // Page253
      0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001 },
    { 0x00000000, 0xfff90000, 0xfef7fe1f, 0x00000f77, 0x00000000,   // Page254
      0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001 }
};


//
//  CType 3 Flag Bits.
//
//  In the interest of reducing our table complexity, we've here a reduced
//  bitfield.  Only those bits currently used by IE4 are returned by
//  GetStringType3Ex().
//

// These are the flags are they are defined in winnls.h
//

// C3_NONSPACING    0x0001
// C3_DIACRITIC     0x0002
// C3_VOWELMARK     0x0004
// C3_SYMBOL        0x0008
// C3_KATAKANA      0x0010
// C3_HIRAGANA      0x0020
// C3_HALFWIDTH     0x0040
// C3_FULLWIDTH     0x0080
// C3_IDEOGRAPH     0x0100
// C3_KASHIDA       0x0200
// C3_LEXICAL       0x0400
// C3_ALPHA         0x8000

// The supported flags are encoded by shifting them to the right 3 bits.

// C3_SYMBOL       0x0001
// C3_KATAKANA     0x0002
// C3_HIRAGANA     0x0004
// C3_HALFWIDTH    0x0008
// C3_FULLWIDTH    0x0010
// C3_IDEOGRAPH    0x0020

// GetStringType3Ex returns the correct Win32 flags NOT the compressed flags.


BOOL 
GetStringType3ExW(
    LPCWSTR lpSrcStr,    // string arg
    int     cchSrc,      // length (or -1)
    LPWORD  lpCharType ) // output buffer
{
    LPCWSTR  lpStop = lpSrcStr + ((cchSrc == -1) ? MAXLONG : cchSrc);

    while (lpSrcStr < lpStop)
    {
        WCHAR wch = *lpSrcStr++;
        WORD wCharType;
        BYTE bPageSub;

        if (!wch && cchSrc == -1)
            break;

        switch (wch & (unsigned int)0xff00)
        {
            case 0x0000:
                wCharType = abType3Page0[wch];         // Page0: 4 values
                break;
            case 0x2000:
                wCharType = abType3Page32[wch & 0xff]; // Page32: 4 values
                break;
            case 0x3000:
                wCharType = abType3Page48[wch & 0xff];  // Page48: 10 values
                break;
            case 0xff00:
                wCharType = abType3Page255[wch & 0xff]; // Page255: 7 values
                break;
            default:
                bPageSub = abType3PageSub[wch>>8];

                if (bPageSub & 0x80)                  // 21 pages have 2 values
                {
                    const struct tagType3DualValue *p = aType3DualValue +
                        (bPageSub & 0x7f);

                    wCharType = (BYTE) p->adwValue[(p->adwBitfield[(wch>>5)&7]
                        >> (wch & 0x1f)) & 1];
                }
                else                                  // 231 pages have 1 value
                {
                    wCharType = bPageSub;
                }
                break;
        }

        *lpCharType++ = wCharType << 3;
    }
    
    return TRUE;
}
