/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    acpievt.c

Abstract:

    This file implements ACPI eventing support within the GPIO class extension
    driver.

    Code that wishes to initialize/de-initialize or enable/disable eventing
    will invoke GpiopManageAcpiEventing(), which will serialize calls to
    GpiopManageAcpiEventingWorker(), which is where the state transitions will
    occur.

    A GPIO bank's ISR will call GpiopProcessAcpiEvents() to handle an _AEI
    event.


Environment:

    Kernel mode

--*/

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"
#include "privdefs.h"
#include "hub.h"

#if defined(EVENT_TRACING)
#include "acpievt.tmh"         // auto-generated by WPP
#endif

//
// -------------------------------------------------------------------- Defines
//

//
// Name for the GPIO ACPI Eventing Information method (_AEI).
//

#define GPIO_AEI_METHOD_AS_ULONG (ULONG)'IEA_'

//
// Name for the ACPI event handler for pins > 0xFF (_EVT).
//

#define GPIO_LONG_PIN_EVENT_METHOD_AS_ULONG (ULONG)'TVE_'


//
// Define the values used to track whether ACPI event was enabled or not for
// a particular descriptor.
//

#define GPIO_ACPI_EVENT_ENABLED (TRUE)
#define GPIO_ACPI_EVENT_NOT_ENABLED (FALSE)

//
// Define the maximum number of retries to be performed if execution of
// _Lxx/_Exx fails for some reason (e.g. lack of resources).
//
//

#define GPIO_ACPI_EVENT_MAXIMUM_RETRIES (4)

typedef enum _ACPI_INTERRUPT_ACTION
{
    AcpiInterruptEnableDisable,

    //
    // Mask/unmask has the following benefits over enable/disable:
    // 
    //   1. Emulated ActiveBoth pins: It ensures that the line stays programmed
    //      looking for the same level when it is unmasked as compared to when
    //      it was masked.
    //     
    //      If it had been disabled instead of masked, the re-enable would have
    //      reprogrammed the controller to look for the asserted level instead,
    //      which would miss any de-assertion of the line.
    //
    //   2. Fewer failure points
    //

    AcpiInterruptMaskUnmask,

    //
    // The regular masking above only disables a pin in hardware.  Deep-masking
    // builds on this by also waiting for almost all interrupt-related activity
    // to cease (debounce state machines/threads).
    //
    // Deep-masking's waiting is similar to that of GpiopFlushPendingPinInterruptActivities(),
    // except that the debounce state machine is not destroyed.
    //
    // Deep-unmasking is identical to the above regular unmasking.
    //

    AcpiInterruptDeepMaskUnmask
} ACPI_INTERRUPT_ACTION;

#define ACPI_ENABLE_DISABLE_STAGE_NONE 0
#define ACPI_ENABLE_DISABLE_STAGE_PINS 1
#define ACPI_ENABLE_DISABLE_STAGE_EVENT_METHODS 2

//
// POP_WATCHDOG_DEFAULT_TIMEOUT is 10 minutes.
//
// Bugcheck before that (in 9 minutes and 50 seconds) since we can provide pin
// information in the bugcheck parameters, rather than requiring the developer
// to scour data from a DRIVER_POWER_STATE_FAILURE (0x9F) dump.  Such scouring
// may be impossible for minidumps.
//
// N.B. It is possible for the user to change "WatchdogSleepTimeout" using the
//      registry, but this 9:50 cannot be adjusted.
//

ULONG GpiopEnableDisableWatchdogSeconds = (9 * 60 + 50);

//
// ----------------------------------------------------------------- Prototypes
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopAllocateAcpiEventResources (
    __in PDEVICE_EXTENSION GpioExtension
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildEdgeLevelPinList (
    __in PDEVICE_EXTENSION GpioExtension,
    __out_ecount(*TotalELEEdgePins + *TotalELEEdgePins +
        *TotalNamespaceEdgePins + *TotalNamespaceLevelPins)
            PULONG *EdgeLevelPinList,

    __out PULONG TotalELEEdgePins,
    __out PULONG TotalELELevelPins,
    __out PULONG TotalNamespaceEdgePins,
    __out PULONG TotalNamespaceLevelPins
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDestroyAcpiEventing (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN Partial
    );

KDEFERRED_ROUTINE GpiopEnableDisableAcpiEventsWatchdog;

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopEnableDisableAcpiEvents (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN Enable,
    __in BOOLEAN AffectNonWakePinsOnly,
    __in ACPI_INTERRUPT_ACTION InterruptAction,
    __in ULONG BugCheckAfterSeconds
    );

PIN_NUMBER
GpiopFindNextEventPinAndClear (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_opt PIN_NUMBER Hint
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopInitiateAcpiEventing (
    __in PDEVICE_EXTENSION GpioExtension
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopParseEventInformation (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_bcount(EventBufferSize) PACPI_EVAL_OUTPUT_BUFFER EventBuffer,
    __in ULONG EventBufferSize,
    __deref_out_bcount(*DescriptorListSize) PIO_RESOURCE_LIST *DescriptorList,
    __out PULONG DescriptorListSize
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopManageAcpiEventing (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ACPI_EVENTING_STATE State,
    __in BOOLEAN Synchronous
    );

VOID
GpiopManageAcpiEventingWorker (
    __in PEVENT_WORK_ITEM_CONTEXT Context
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopPrepareForAcpiEvents(
    __in PDEVICE_EXTENSION GpioExtension
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopFlushPendingAcpiEvents (
    __in PDEVICE_EXTENSION GpioExtension
    );

EVT_WDF_WORKITEM GpiopManageAcpiEventingWorkerWrapper;

EVT_WDF_DPC GpiopProcessAcpiEventsDpc;

_IRQL_requires_same_
_IRQL_requires_max_(DISPATCH_LEVEL)
VOID
GpiopProcessAcpiEventsCommon(
    _In_ WDFDPC Dpc
    );

WORKER_THREAD_ROUTINE GpiopProcessAcpiEventsWorker;

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopValidateMethodsForEvents (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_bcount(DescriptorListSize) PIO_RESOURCE_LIST DescriptorList,
    __in ULONG DescriptorListSize
    );

_Function_class_(POWER_SETTING_CALLBACK)
_IRQL_requires_same_
NTSTATUS
GpiopPowerAcpiEventingCallback (
    _In_ LPCGUID SettingGuid,
    _In_reads_bytes_(ValueLength) PVOID Value,
    _In_ ULONG ValueLength,
    _Inout_opt_ PVOID Context
    );

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text(PAGE, GpiopAllocateAcpiEventResources)
#pragma alloc_text(PAGE, GpiopBuildEdgeLevelPinList)
#pragma alloc_text(PAGE, GpiopDestroyAcpiEventing)
#pragma alloc_text(PAGE, GpiopInitiateAcpiEventing)
#pragma alloc_text(PAGE, GpiopIsAcpiEventInterrupt)
#pragma alloc_text(PAGE, GpiopParseEventInformation)
#pragma alloc_text(PAGE, GpiopValidateEventMethodForPin)
#pragma alloc_text(PAGE, GpiopValidateMethodsForEvents)

//
// ------------------------------------------------------------------ Functions
//

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopManageAcpiEventing (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ACPI_EVENTING_STATE State,
    __in BOOLEAN Synchronous
    )

/*++

Routine Description:

    This function initiates ACPI eventing on the GPIO controller. As the
    process involves evaluating various methods, sending IOCTL to Resource
    hub etc., it is done in a separate worker thread.

    It waits for any previous call / state change to complete before moving on
    to the one that is being requested now.


    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    Device - Supplies a handle to the framework device object.

    State - Supplies the state indicating ACPI eventing steps to perform.

    Synchronous - Supplies a flag indicating whether the operation for
        the state should be performed synchronously or asynchronously.

Return Value:

    NTSTATUS code.

--*/

{

    WDF_OBJECT_ATTRIBUTES Attributes;
    PEVENT_WORK_ITEM_CONTEXT Context;
    EVENT_WORK_ITEM_CONTEXT LocalContext;
    NTSTATUS Status;
    WDFWORKITEM WorkItem;
    WDF_WORKITEM_CONFIG WorkItemConfiguration;

    Context = NULL;
    WorkItem = NULL;

    //
    // This should never be called on the hub device extension.
    //

    GPIO_ASSERT(GPIO_IS_HUB_DEVICE(GpioExtension->Device) == FALSE);

    //
    // Create a workitem to process ACPI events. The WDF device object needs
    // to be passed as part of the workitem context. Since this is the same
    // information required when processing events, the same context
    // structure is being reused here.
    //

    WDF_OBJECT_ATTRIBUTES_INIT(&Attributes);
    WDF_OBJECT_ATTRIBUTES_SET_CONTEXT_TYPE(&Attributes,
                                           EVENT_WORK_ITEM_CONTEXT);

    Attributes.ParentObject = GpioExtension->Device;

    //
    // Initialize the handler routine and create a new workitem.
    //

    WDF_WORKITEM_CONFIG_INIT(&WorkItemConfiguration,
                             GpiopManageAcpiEventingWorkerWrapper);

    //
    // Disable automatic serialization by the framework for the worker thread.
    // The parent device object is being serialized at device level (i.e.,
    // WdfSynchronizationScopeDevice), and the framework requires it be
    // passive level (i.e., WdfExecutionLevelPassive) if automatic
    // synchronization is desired.
    //

    WorkItemConfiguration.AutomaticSerialization = FALSE;

    //
    // Wait for the previous stage to complete.
    //

    Status = KeWaitForSingleObject(&GpioExtension->EventData.MutexEvent,
                                   Executive,
                                   KernelMode,
                                   FALSE,
                                   NULL);

    NT_ASSERT(NT_SUCCESS(Status) != FALSE);

    //
    // If the resources could be successfully allocated, reset the previous
    // state for enable state even if it was failed. This allows subsequent
    // enables to still be attempted (this is possible if the GPIO controller
    // was stopped and then later re-started, and the first time around,
    // enabling the events failed for some reason).
    //

    if ((State == AcpiEventStateEnable) &&
        (GpioExtension->EventData.ResourcesAllocated != FALSE)) {

        GpioExtension->EventData.PreviousStateFailed = FALSE;
    }

    //
    // If there are no events to be processed, then force the enable/disable
    // step to be done synchronously. This eliminates the need to queue a
    // worker thread. The routine handling enable/disable will bail out
    // immediately but the worker needs to be called such that the mutex event
    // is reset appropriately for the next stage.
    //

    if ((State != AcpiEventStateDestroy) &&
        (GpioExtension->EventData.PreviousStateFailed != FALSE)) {

        Synchronous = TRUE;
    }

    //
    // Create the work item and queue it. If the workitem cannot be created
    // for some reason, just call the worker routine synchronously.
    //

    if (Synchronous == FALSE) {
        Status = WdfWorkItemCreate(&WorkItemConfiguration,
                                   &Attributes,
                                   &WorkItem);

        if (NT_SUCCESS(Status)) {
            Context = GpiopGetEventWorkItemContext(WorkItem);

        } else {
            Synchronous = TRUE;
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_ACPIEVT,
                        "Failed to allocate work item to initialize ACPI "
                        "eventing! Status = %#x\n",
                        Status);
        }
    }

    if (Context == NULL) {
        Context = &LocalContext;
    }

    //
    // Initialize the context to be supplied to the workitem handler.
    //

    RtlZeroMemory(Context, sizeof(EVENT_WORK_ITEM_CONTEXT));
    Context->Device = GpioExtension->Device;
    Context->State = State;
    Context->Synchronous = Synchronous;

    //
    // If the operation is to be performed synchronously, the directly
    // invoke the worker routine. Otherwise, queue a workitem to run the
    // worker routine.
    //

    if (Synchronous != FALSE) {
        GpiopManageAcpiEventingWorker(Context);

    } else {
        WdfWorkItemEnqueue(WorkItem);
    }

    return STATUS_SUCCESS;
}

VOID
GpiopManageAcpiEventingWorkerWrapper (
    __in WDFWORKITEM WorkItem
    )

/*++

Routine Description:

    This function is a worker routine that invokes the appropriate handler
    based on the state in the workitem context.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    WorkItem -  Supplies a handle to the workitem supplying the context.

Return Value:

    None.

--*/

{

    PEVENT_WORK_ITEM_CONTEXT Context;

    Context = GpiopGetEventWorkItemContext(WorkItem);
    GpiopManageAcpiEventingWorker(Context);

    //
    // Delete the work item as it is no longer required.
    //

    WdfObjectDelete(WorkItem);
    return;
}

VOID
GpiopManageAcpiEventingWorker (
    __in PEVENT_WORK_ITEM_CONTEXT Context
    )

/*++

Routine Description:

    This function is a worker routine that invokes the appropriate handler
    based on the state in the workitem context.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    WorkItem -  Supplies a handle to the workitem supplying the context.

    Context - Supplies a pointer to the workitem context.

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    BOOLEAN Enable;
    PDEVICE_EXTENSION GpioExtension;
    BOOLEAN Partial;
    NTSTATUS Status;

    Device = Context->Device;
    Enable = FALSE;
    GpioExtension = GpioClxGetDeviceExtension(Device);
    Partial = FALSE;

    if ((Context->State != AcpiEventStateDestroy) &&
        (GpioExtension->EventData.PreviousStateFailed != FALSE)) {

        Status = STATUS_SUCCESS;
        goto SetEvent;
    }

    //
    // Initialize the handler routine and create a new workitem.
    //

    switch (Context->State) {

    case AcpiEventStateInitialize:
        Status = GpiopInitiateAcpiEventing(GpioExtension);
        break;

    case AcpiEventStateAllocateResources:
        Status = GpiopAllocateAcpiEventResources(GpioExtension);
        break;

    case AcpiEventStateEnable:
        Enable = TRUE;
        __fallthrough;

    case AcpiEventStateDisable:
        Status = GpiopEnableDisableAcpiEvents(GpioExtension, Enable, FALSE, AcpiInterruptEnableDisable, 0);
        break;

    case AcpiEventStateEnableForConnectedStandby:
        Enable = TRUE;
        __fallthrough;

    case AcpiEventStateDisableForConnectedStandby:
        Status = GpiopEnableDisableAcpiEvents(GpioExtension, Enable, TRUE, AcpiInterruptMaskUnmask, 0);
        break;

    case AcpiEventStateDeepUnmaskForSx:
        Enable = TRUE;
        Status = GpiopEnableDisableAcpiEvents(GpioExtension, Enable, FALSE, AcpiInterruptDeepMaskUnmask, 0);
        break;

    case AcpiEventStateDeepMaskForSx:
        Enable = FALSE;
        Status = GpiopEnableDisableAcpiEvents(GpioExtension, Enable, FALSE, AcpiInterruptDeepMaskUnmask, GpiopEnableDisableWatchdogSeconds);
        break;

    case AcpiEventStateReleaseResources:
        Partial = TRUE;
        __fallthrough;

    case AcpiEventStateDestroy:
        GpiopDestroyAcpiEventing(GpioExtension, Partial);
        Status = STATUS_SUCCESS;
        break;

    default:

        GPIO_ASSERT(FALSE);

        Status = STATUS_SUCCESS;
    }

    if (!NT_SUCCESS(Status)) {
        GpioExtension->EventData.PreviousStateFailed = TRUE;
    }

SetEvent:

    //
    // Set the event for the next state to begin.
    //

    KeSetEvent(&GpioExtension->EventData.MutexEvent, IO_NO_INCREMENT, FALSE);

    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopInitiateAcpiEventing (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This function is a worker function that performs all the steps
    necessary to enable ACPI eventing on the GPIO controller. The steps it
    takes are:
    1. Evaluate _AEI method to fetch the resource descriptors
    2. Validate all the descriptors
    3. Communicate with the Resource hub to translate the BIOS resources
       to NT (interrupt) resources

    Note the actual enabling of the interrupt is done in a different
    phase after the controller has started.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    PIO_RESOURCE_LIST DescriptorList;
    ULONG DescriptorListSize;
    WDFDEVICE Device;
    PGPIO_ACPI_EVENT_DATA EventData;
    PVOID EventOutputBuffer;
    ULONG EventOutputBufferSize;
    NTSTATUS Status;

    PAGED_CODE();

    Device = GpioExtension->Device;
    DescriptorList = NULL;
    EventOutputBuffer = NULL;

    //
    // Send an IRP to ACPI to evaluate the _AEI method. The method is required
    // to be defined directly under the namespace of the GPIO controller.
    //

    Status = GpioUtilEvaluateAcpiMethod(Device,
                                        GPIO_AEI_METHOD_AS_ULONG,
                                        NULL,
                                        &EventOutputBuffer,
                                        &EventOutputBufferSize);

    if (Status == STATUS_OBJECT_NAME_NOT_FOUND) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_ACPIEVT,
                    "_AEI method does not exist. ACPI eventing will "
                    "not be enabled! Extn = 0x%p, Status = %#x\n",
                    GpioExtension,
                    Status);

        goto InitiateAcpiEventingEnd;

    } else if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "Failed to evaluate _AEI method. ACPI eventing will "
                    "not be enabled! Status = %#x\n",
                    Status);

        goto InitiateAcpiEventingEnd;
    }

    //
    // If the method evaluation succeeded but it didn't return any buffer,
    // then treat it as if _AEI was not present.
    //

    if (NT_SUCCESS(Status) && (EventOutputBuffer == NULL)) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_ACPIEVT,
                    "_AEI method evaluation didn't return any buffer!"
                    "Skipping enabling of ACPI eventing...\n");

        Status = STATUS_UNSUCCESSFUL;
        goto InitiateAcpiEventingEnd;
    }

    //
    // Parse and validate the information returned from the firmware.
    //

    Status = GpiopParseEventInformation(GpioExtension,
                                        EventOutputBuffer,
                                        EventOutputBufferSize,
                                        &DescriptorList,
                                        &DescriptorListSize);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "Error parsing _AEI method output buffer! "
                    "Status = %#x\n",
                    Status);

        goto InitiateAcpiEventingEnd;
    }

    //
    // Free the event buffer as it is no longer required.
    //

    GPIO_FREE_POOL(EventOutputBuffer);
    EventOutputBuffer = NULL;

    //
    // If there are no valid GPIO interrupt descriptors returned by _AEI method,
    // skip rest of the processing.
    //

    if (DescriptorList == NULL) {
        goto InitiateAcpiEventingEnd;
    }

    EventData = &GpioExtension->EventData;
    Status = GpiopBuildEdgeLevelPinList(GpioExtension,
                                        &EventData->EdgeLevelPins,
                                        &EventData->ELEEdgePinCount,
                                        &EventData->ELELevelPinCount,
                                        &EventData->NamespaceEdgePinCount,
                                        &EventData->NamespaceLevelPinCount);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_ACPIEVT,
                    "Error trying to build event handler list! "
                    "ACPI eventing will not be enabled! "
                    "Extn = 0x%p, Status = %#x\n",
                    GpioExtension,
                    Status);

        goto InitiateAcpiEventingEnd;
    }

    //
    // Validate that there is a corresponding _Lxx/_Exx or _EVT method defined
    // for every pin to be used as an event.
    //

    Status = GpiopValidateMethodsForEvents(GpioExtension,
                                           DescriptorList,
                                           DescriptorListSize);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "Failed to validate descriptors in _AEI with namespace "
                    "methods! Status = %#x\n",
                    Status);

        goto InitiateAcpiEventingEnd;
    }

    GpioExtension->EventData.DescriptorList = DescriptorList;
    GpioExtension->EventData.DescriptorListSize = DescriptorListSize;

    //
    // Register for PDC notification to disconnect AEIs upon engaging IR.
    //

    Status = PoRegisterPowerSettingCallback(
                NULL,
                &GUID_PDC_IDLE_RESILIENCY_ENGAGED,
                GpiopPowerAcpiEventingCallback,
                GpioExtension,
                &GpioExtension->EventData.PowerSettingCallbackHandle);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "Failed to register power setting callback for _AEI "
                    "methods! Status = %#x\n",
                    Status);

        GpioExtension->EventData.PowerSettingCallbackHandle = NULL;
        goto InitiateAcpiEventingEnd;
    }

InitiateAcpiEventingEnd:
    if (EventOutputBuffer != NULL) {
        GPIO_FREE_POOL(EventOutputBuffer);
    }

    if (!NT_SUCCESS(Status) && (DescriptorList != NULL)) {
        GPIO_FREE_POOL(DescriptorList);

        GpioExtension->EventData.DescriptorList = NULL;
        GpioExtension->EventData.DescriptorListSize = 0;
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopAllocateAcpiEventResources (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine pre-allocates the memory resources as well as a DPC and a
    worker item necessary for event processing. This reduces the chances for
    runtime failures during ACPI event processing due to lack of resources.


Arguments:

    Device - Supplies a handle to the framework device object.

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    WDF_OBJECT_ATTRIBUTES Attributes;
    PVOID Buffer;
    WDF_DPC_CONFIG DpcConfiguration;
    PGPIO_ACPI_EVENT_DATA EventsData;
    ULONG Size;
    NTSTATUS Status;
    PIN_NUMBER TotalPins;

    PAGED_CODE();

    //
    // Allocate a bitmap buffer if necessary. Bitmap buffers are only required
    // if pins exceed long pin threshold. The size of the buffer should be the
    // same as what is required for other interrupt-related buffers.
    //

    TotalPins = GpioExtension->TotalPins;
    Size = ALIGN_RANGE_UP(TotalPins, (sizeof(ULONG) * 8)) / 8;
    Buffer = GPIO_ALLOCATE_POOL(NonPagedPoolNx, Size);
    if (Buffer == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "%s: Failed to allocate memory for eventing bitmap "
                    "buffer!\n",
                    __FUNCTION__);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto AllocateAcpiEventResourcesEnd;
    }

    RtlZeroMemory(Buffer, Size);
    RtlInitializeBitMap(&GpioExtension->EventData.PendingEventsBitmap,
                        Buffer,
                        TotalPins);

    //
    // Create the DPC to process ACPI events.
    //
    // For serially-accessible GPIOs, the DPC itself is never queued. However,
    // the DPC object is used (to get to the GPIO extension).
    //

    EventsData = &GpioExtension->EventData;
    WDF_DPC_CONFIG_INIT(&DpcConfiguration, GpiopProcessAcpiEventsDpc);
    DpcConfiguration.AutomaticSerialization = FALSE;
    WDF_OBJECT_ATTRIBUTES_INIT(&Attributes);
    Attributes.ParentObject = GpioExtension->Device;
    Status = WdfDpcCreate(&DpcConfiguration,
                          &Attributes,
                          &EventsData->EventDpc);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "%s: WdfDpcCreate() failed! Status = %#x\n ",
                    __FUNCTION__,
                    Status);

        goto AllocateAcpiEventResourcesEnd;
    }

    //
    // Initialize a WDM workitem to process ACPI events.
    //
    // N.B. This routine creates a WDM workitem instead of a WDF one. This is
    //      because there may be scenarios where the system worker thread
    //      pool may not be available and ACPI events must be processed to
    //      guarantee forward progress (e.g. it is wake event).
    //
    //      An EX work-item (WORK_QUEUE_ITEM) is used instead of the IO_WORKITEM
    //      as the IO one does not allowed it to be re-used as an emergency
    //      work item (the worker routine and parameter are fixed).
    //

    ExInitializeWorkItem(&GpioExtension->EventData.EventWdmWorkItem,
                         GpiopProcessAcpiEventsWorker,
                         GpioExtension);

    GpioExtension->EventData.ResourcesAllocated = TRUE;

    //
    // N.B. On failure, the event bitmap buffer will be deleted when the device
    //      extension is deleted.
    //

AllocateAcpiEventResourcesEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildEdgeLevelPinList (
    __in PDEVICE_EXTENSION GpioExtension,
    __out_ecount(*TotalELEEdgePins + *TotalELEEdgePins +
        *TotalNamespaceEdgePins + *TotalNamespaceLevelPins)
            PULONG *EdgeLevelPinList,

    __out PULONG TotalELEEdgePins,
    __out PULONG TotalELELevelPins,
    __out PULONG TotalNamespaceEdgePins,
    __out PULONG TotalNamespaceLevelPins
    )

/*++

Routine Description:

    This function queries the ACPI namespace and builds a list of pins for
    which there are event handlers (_Exx, _Lxx, _EVT) defined in the namespace.


Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    EdgeLevelPinList - Supplies a pointer that receives the buffer that
        contains the list of pins. The caller is responsible for freeing this
        buffer.

    TotalELEEdgePins - Supplies a pointer that receives the number of
        edge-triggered pins handled by _EVT method.

    TotalELELevelPins - Supplies a pointer that receives the number of
        level-triggered pins handled by _EVT method.

    TotalNamespaceEdgePins - Supplies a pointer that receives the number of
        edge-triggered pins handled by _Exx methods.

    TotalNamespaceLevelPins - Supplies a pointer that receives the number of
        level-triggered pins handled by _Lxx methods.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG ChildEdgePinCount;
    ULONG ChildLevelPinCount;
    PULONG EdgeLevelPins;
    ULONG EdgePinCount;
    PACPI_EVAL_OUTPUT_BUFFER ELEBuffer;
    ULONG ELEBufferSize;
    ULONG ELEEdgePinCount;
    ULONG ELELevelPinCount;
    PACPI_ENUM_CHILDREN_OUTPUT_BUFFER EnumChildrenBuffer;
    ULONG EnumChildrenBufferSize;
    ULONG LevelPinCount;
    ULONG StartIndex;
    NTSTATUS Status;
    ULONG TotalPins;

    PAGED_CODE();

    EdgeLevelPins = NULL;
    EdgePinCount =  0;
    ELEBuffer = NULL;
    ELEBufferSize = 0;
    ELEEdgePinCount = 0;
    ELELevelPinCount = 0;
    EnumChildrenBuffer = NULL;
    EnumChildrenBufferSize = 0;
    LevelPinCount = 0;

    //
    // Send an IRP to ACPI to enumerate the children that are defined directly
    // under this controller's namespace.
    //

    Status = GpioUtilEnumerateNamespaceChildren(GpioExtension,
                                                &EnumChildrenBuffer,
                                                &EnumChildrenBufferSize);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "%s: Failed to enumerate ACPI namespace children! "
                    "Extn = %p, Status = %#x\n",
                    __FUNCTION__,
                    GpioExtension,
                    Status);

        goto BuildEdgeLevelPinListEnd;
    }

    //
    // Check whether _EVT is defined in the namespace or not.
    //

    if (EnumChildrenBuffer != NULL) {
        Status = GpioUtilIsChildObjectDefined(
                    EnumChildrenBuffer,
                    EnumChildrenBufferSize,
                    GPIO_LONG_PIN_EVENT_METHOD_AS_ULONG);

        if (NT_SUCCESS(Status)) {
            GpioExtension->EventData.EVTMethodDefined = TRUE;

            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_ACPIEVT,
                        "%s: _EVT() is defined in namespace. Extn = %p\n",
                        __FUNCTION__,
                        GpioExtension);

        } else if (Status == STATUS_OBJECT_NAME_NOT_FOUND) {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_ACPIEVT,
                        "%s: _EVT() is NOT defined in namespace. Extn = %p\n",
                        __FUNCTION__,
                        GpioExtension);

            Status = STATUS_SUCCESS;
            __fallthrough;

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_ACPIEVT,
                        "%s: Failed to enumerate ACPI namespace children! "
                        "Extn = %p, Status = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        Status);

            goto BuildEdgeLevelPinListEnd;
        }
    }


    //
    // Iterate through all the namespace children.
    //

    Status = GpioUtilParseNamespaceEventPins(EnumChildrenBuffer,
                                             EnumChildrenBufferSize,
                                             NULL,
                                             0,
                                             &ChildEdgePinCount,
                                             NULL,
                                             0,
                                             &ChildLevelPinCount);

    if (!NT_SUCCESS(Status) && (Status != STATUS_BUFFER_TOO_SMALL)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "%s: Failed to parse namespace children list! "
                    "Extn = %p, Status = %#x\n",
                    __FUNCTION__,
                    GpioExtension,
                    Status);

        goto BuildEdgeLevelPinListEnd;
    }


    EdgePinCount = ChildEdgePinCount + ELEEdgePinCount;
    LevelPinCount = ChildLevelPinCount + ELELevelPinCount;
    TotalPins = EdgePinCount + LevelPinCount;
    if (TotalPins == 0x0) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_ACPIEVT,
                    "%s: No namespace _Lxx/_Exx children present!",
                    __FUNCTION__);

        Status = STATUS_SUCCESS;
        goto SetOutputParametersEnd;
    }

    //
    // Allocate a buffer of adequate size to hold all the edge+level pin list.
    //

    EdgeLevelPins = GPIO_ALLOCATE_POOL(PagedPool, TotalPins * sizeof(ULONG));
    if (EdgeLevelPins == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "%s: Failed to allocate buffer for combined edge/level"
                    "pin list! Extn = %p\n",
                    __FUNCTION__,
                    GpioExtension);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto BuildEdgeLevelPinListEnd;
    }

    RtlZeroMemory(EdgeLevelPins, TotalPins * sizeof(ULONG));

    //
    // Iterate through all the namespace children.
    //

    if ((ChildEdgePinCount > 0) || (ChildLevelPinCount > 0)) {
        Status = GpioUtilParseNamespaceEventPins(
                    EnumChildrenBuffer,
                    EnumChildrenBufferSize,
                    &EdgeLevelPins[0],
                    ChildEdgePinCount,
                    &ChildEdgePinCount,
                    &EdgeLevelPins[EdgePinCount],
                    ChildLevelPinCount,
                    &ChildLevelPinCount);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_ACPIEVT,
                        "%s: Failed to parse namespace children list! "
                        "Extn = %p, Status = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        Status);

            GPIO_FREE_POOL(EdgeLevelPins);
            EdgeLevelPins = NULL;

            goto BuildEdgeLevelPinListEnd;
        }
    }


SetOutputParametersEnd:
    *EdgeLevelPinList = EdgeLevelPins;
    *TotalNamespaceEdgePins = ChildEdgePinCount;
    *TotalNamespaceLevelPins = ChildLevelPinCount;
    *TotalELEEdgePins = ELEEdgePinCount;
    *TotalELELevelPins = ELELevelPinCount;

BuildEdgeLevelPinListEnd:
    if (ELEBuffer != NULL) {
        GPIO_FREE_POOL(ELEBuffer);
    }

    if (EnumChildrenBuffer != NULL) {
        GPIO_FREE_POOL(EnumChildrenBuffer);
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDestroyAcpiEventing (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN Partial
    )

/*++

Routine Description:

    This routine destroys all the buffers allocated for ACPI eventing.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    PRTL_BITMAP EventsBitmap;

    PAGED_CODE();

    EventsBitmap = &GpioExtension->EventData.PendingEventsBitmap;
    if (EventsBitmap->Buffer != NULL) {
        GPIO_FREE_POOL(EventsBitmap->Buffer);
        EventsBitmap->Buffer = NULL;
    }

    if (GpioExtension->EventData.EdgeLevelPins != NULL) {
        GPIO_FREE_POOL(GpioExtension->EventData.EdgeLevelPins);
        GpioExtension->EventData.EdgeLevelPins = NULL;
    }

    if (GpioExtension->EventData.EventDpc != NULL) {
        WdfObjectDelete(GpioExtension->EventData.EventDpc);
        GpioExtension->EventData.EventDpc = NULL;
    }

    //
    // Unregister from Po notification
    //

    if (GpioExtension->EventData.PowerSettingCallbackHandle != NULL) {
        PoUnregisterPowerSettingCallback(
            GpioExtension->EventData.PowerSettingCallbackHandle);

        GpioExtension->EventData.PowerSettingCallbackHandle = NULL;
    }

    //
    // If the device is going away, then also delete the descriptor list.
    //

    if ((Partial == FALSE) &&
        (GpioExtension->EventData.DescriptorList != NULL)) {

        GPIO_FREE_POOL(GpioExtension->EventData.DescriptorList);
        GpioExtension->EventData.DescriptorList = NULL;
    }

    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopParseEventInformation (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_bcount(EventBufferSize) PACPI_EVAL_OUTPUT_BUFFER EventBuffer,
    __in ULONG EventBufferSize,
    __deref_out_bcount(*DescriptorListSize) PIO_RESOURCE_LIST *DescriptorList,
    __out PULONG DescriptorListSize
    )

/*++

Routine Description:

    This function communicates with the resource hub to parse the resources
    specified within the _AEI object. All resources must be GPIO interrupt
    resources; other types of resources are ignored.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    EventBuffer - Supplies a buffer containing the output from _AEI method
        evaluation.

    EventBufferSize - Supplies the size of the event buffer.

    DescriptorList - Supplies a pointer that receives a buffer containing the
        translated descriptors. The buffer is allocated by the callee and
        must be freed by the caller once it is done using it.

    DescriptorListSize - Supplies a pointer that receives the size of the
        buffer allocated for output.

Return Value:

    NTSTATUS code.

--*/

{

    PACPI_METHOD_ARGUMENT Argument;
    WDF_MEMORY_DESCRIPTOR InputDescriptor;
    WDFIOTARGET IoTarget;
    IO_RESOURCE_LIST LocalOutputBuffer;
    PIO_RESOURCE_LIST OutputBuffer;
    ULONG OutputBufferSize;
    WDF_MEMORY_DESCRIPTOR OutputDescriptor;
    ULONG_PTR SizeReturned;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(EventBufferSize);

    PAGED_CODE();

    //
    // Validate the input buffer length supplied.
    //

    OutputBuffer = NULL;
    if (EventBufferSize < sizeof(ACPI_METHOD_ARGUMENT)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Argument buffer size is not sufficient!\n",
                    __FUNCTION__);

        Status = STATUS_ACPI_INVALID_DATA;
        goto ParseEventInformationEnd;
    }

    //
    // The input buffer (i.e., the output buffer from evaluating the _AEI
    // method) should have one argument of buffer type. Ignore any other data
    // that is passed. Additional data is not treated as fatal to allow
    // ACPI eventing to work on pins/descriptors that are described correctly.
    //

    Argument = &EventBuffer->Argument[0];
    if (Argument->Type != ACPI_METHOD_ARGUMENT_BUFFER) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: First argument is not a buffer argument!\n",
                    __FUNCTION__);

        Status = STATUS_ACPI_INVALID_DATA;
        goto ParseEventInformationEnd;
    }

    //
    // Get the default IO target to send the translation IOCTL.
    //

    IoTarget = WdfDeviceGetIoTarget(GpioExtension->Device);

    //
    // Initialize the input and output buffers and corresponding WDF
    // descriptors.
    //

    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&InputDescriptor,
                                      (PVOID)Argument->Data,
                                      Argument->DataLength);

    RtlZeroMemory(&LocalOutputBuffer, sizeof(LocalOutputBuffer));
    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&OutputDescriptor,
                                      (PVOID)&LocalOutputBuffer,
                                      sizeof(LocalOutputBuffer));

    //
    // Send the IOCTL the first time to determine the size of the output
    // buffer needed. Note the input buffer may be sufficient if there is only
    // GPIO interrupt descriptor or there are no valid descriptors returned
    // by _AEI method.
    //

    SizeReturned = 0;
    Status = WdfIoTargetSendIoctlSynchronously(IoTarget,
                                               NULL,
                                               IOCTL_ACPI_TRANSLATE_BIOS_RESOURCES,
                                               &InputDescriptor,
                                               &OutputDescriptor,
                                               NULL,
                                               &SizeReturned);

    //
    // If the translation succeeded, then it implies either no descriptors
    // were specified or none of them were valid. In such case, treat this as
    // as case of missing _AEI object and skip processing.
    //
    // If it failed with anything other than STATUS_BUFFER_OVERFLOW, treat it
    // as as fatal failure.
    //

    if (!NT_SUCCESS(Status) && (Status != STATUS_BUFFER_OVERFLOW)) {
        goto ParseEventInformationEnd;
    }

    //
    // If there are no descriptors (atleast ones that are valid) returned by
    // the _AEI object, then simply bail out.
    //

    if (LocalOutputBuffer.Count == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_ACPIEVT,
                    "%s: No valid descriptors returned by _AEI!\n",
                    __FUNCTION__);

        *DescriptorList = NULL;
        *DescriptorListSize = 0;
        Status = STATUS_SUCCESS;
        goto ParseEventInformationEnd;
    }

    //
    // Allocate the output buffer and initialize the WDF descriptor.
    //

    OutputBufferSize = FIELD_OFFSET(IO_RESOURCE_LIST, Descriptors) +
        (LocalOutputBuffer.Count * sizeof(IO_RESOURCE_DESCRIPTOR));

    OutputBuffer = GPIO_ALLOCATE_POOL(PagedPool, OutputBufferSize);
    if (OutputBuffer == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_ACPIEVT,
                    "%s: Failed to allocate memory for translation output "
                    "buffer! \n",
                    __FUNCTION__);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto ParseEventInformationEnd;
    }

    //
    // If there is only GPIO interrupt descriptor in the input buffer, then
    // there is no need to dispatch the request a second time.
    //

    if (SizeReturned == sizeof(LocalOutputBuffer)) {
        RtlCopyMemory(OutputBuffer, &LocalOutputBuffer, SizeReturned);

    } else {
        WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&OutputDescriptor,
                                          (PVOID)OutputBuffer,
                                          OutputBufferSize);

        //
        // Send the IOCTL the second time to get the translated descriptors.
        //

        Status = WdfIoTargetSendIoctlSynchronously(
                     IoTarget,
                     NULL,
                     IOCTL_ACPI_TRANSLATE_BIOS_RESOURCES,
                     &InputDescriptor,
                     &OutputDescriptor,
                     NULL,
                     &SizeReturned);
    }

    if (NT_SUCCESS(Status)) {

        GPIO_ASSERT(SizeReturned == OutputBufferSize);

        *DescriptorList = OutputBuffer;
        *DescriptorListSize = OutputBufferSize;
    }

ParseEventInformationEnd:
    if (!NT_SUCCESS(Status) && (OutputBuffer != NULL)) {
        GPIO_FREE_POOL(OutputBuffer);
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopWaitForCompletionOfMaskedEventingPin (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Gsiv
    )

/*++

Routine Description:

    This routine wraps GpiopWaitForStateMachineCompletionOfDebouncedMaskedPin().
    However, the current routine does not require that the pin be
    software-debounced.
         
Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Gsiv - Supplies the GSIV of the interrupt pin.

Return Value:

    TRUE if the caller should subsequently call KeFlushQueuedDpcs() so that the
    pin's debounce or noise timer DPC will complete.

--*/

{

    PGPIO_BANK_ENTRY GpioBank;
    BOOLEAN IsDebounced;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER PinNumber;

    GpioBank = NULL;
    PinNumber = INVALID_PIN_NUMBER;
    PinInformation =
        GpiopGetPinEntryFromVirq(
            GpioExtension,
            Gsiv,
            &GpioBank,
            NULL,
            &PinNumber);

    GPIO_ASSERT(PinInformation != NULL);
    GPIO_ASSERT(GpioBank != NULL);
    GPIO_ASSERT(PinNumber != INVALID_PIN_NUMBER);

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    GPIO_ASSERT(PinInformation->Mode.Interrupt == 1);

    IsDebounced = (PinInformation->Mode.EmulateDebounce == 1);

    GPIO_RELEASE_BANK_LOCK(GpioBank);
    
    if (IsDebounced == FALSE) {
        return FALSE;
    }

    //
    // If the pin asserted before the mask committed to hardware, it is
    // possible that the execution of the parent bank's ISR could have been
    // delayed until after the mask committed.  Luckily, the bank ISR will
    // disregard that pin (see GpiopUpdateStatusAndMaskInformation()).  This
    // means that, at this point, the pin's ISR will not fire again.
    //
    // However, if the bank ISR fired before the above mask, it
    // is possible that the pin is still going through the debounce
    // process.  In that case, wait for the debounce state machine
    // to complete.
    //

    GpiopWaitForStateMachineCompletionOfDebouncedMaskedPin(
        GpioBank,
        PinInformation,
        PinNumber,
        FALSE);  // Don't flush timer DPC yet

    //
    // Request that the caller flush the timer DPC in case the caller can
    // batch multiple flushes.
    //

    return TRUE;
}

_Use_decl_annotations_
VOID
GpiopEnableDisableAcpiEventsWatchdog (
    PKDPC Dpc,
    PVOID Context,
    PVOID Argument1,
    PVOID Argument2
    )

/*++

Routine Description:

    This routine bugchecks the system.  It is called when the watchdog DPC
    expires while GpiopEnableDisableAcpiEvents() is running.

Arguments:

    Dpc - Supplies a pointer to the DPC.

    Context - Supplies a pointer to the GPIO's device extension.

    Argument1 - Unused.

    Argument2 - Unused.

Return Value:

    None.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    ULONG Stage;
    ULONG PinIdentifier;

    UNREFERENCED_PARAMETER(Dpc);
    UNREFERENCED_PARAMETER(Argument1);
    UNREFERENCED_PARAMETER(Argument2);

    GpioExtension = (PDEVICE_EXTENSION)Context;
    Stage = InterlockedOr(&GpioExtension->EventData.EnableDisableStage, 0);
    switch(Stage) {
    default:
    case ACPI_ENABLE_DISABLE_STAGE_NONE:
        PinIdentifier = 0;
        break;

    case ACPI_ENABLE_DISABLE_STAGE_PINS:
        PinIdentifier = (ULONG)InterlockedOr(&GpioExtension->EventData.EnableDisableVector, 0);
        break;

    case ACPI_ENABLE_DISABLE_STAGE_EVENT_METHODS:
        PinIdentifier = (ULONG)InterlockedOr(&GpioExtension->EventData.LastAcpiMethod, 0);
        break;
    }

    KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                 ENABLE_DISABLE_TIMEOUT,
                 (ULONG_PTR)GpioExtension,
                 Stage,
                 PinIdentifier);

    return;
}

C_ASSERT((GPIO_ACPI_EVENT_NOT_ENABLED == FALSE) &&
         (GPIO_ACPI_EVENT_ENABLED == TRUE));

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopEnableDisableAcpiEvents (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN Enable,
    __in BOOLEAN AffectNonWakePinsOnly,
    __in ACPI_INTERRUPT_ACTION InterruptAction,
    __in ULONG BugCheckAfterSeconds
    )

/*++

Routine Description:

    This routine enables/disables pins specified within the _AEI object to be
    used for eventing. It sends IRPs to the GPIO controller (i.e. IRPs to
    itself) to get the pins specified within the _AEI object
    connected/disconnected for interrupts. Since this routine runs in a worker
    thread context, the IRPs are necessary to ensure interrupt enable is
    properly synchronized with the power state.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies. It may
    also be called before/after connected standby and spinning up the storage
    device would slow down the transition.


Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Enable - Supplies a flag indicating whether events have to be enabled
        (TRUE) or disabled (FALSE).

    AffectNonWakePinsOnly - Supplies a flag indicating whether only non-wake
        pins should be affected.  This is set to TRUE for connected standby.

        If this routine is being asked to disable all pins (i.e. both Enable
        and AffectNonWakePinsOnly are set to FALSE), this will block waiting
        for any eventing activity to complete.

    InterruptAction - Supplies the action to be applied to the affected pins.
        This is set to InterruptActionMaskUnmask for connected standby.

    BugCheckAfterSeconds - Supplies the time in seconds that this routine is
        allowed to run for. After that time, a bugcheck will be initiated.
        Set this to 0 if there is no time limit.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN WaitForDebounceTimerDpcs;
    ULONG Count;
    PIO_RESOURCE_DESCRIPTOR Descriptor;
    PIO_RESOURCE_LIST DescriptorList;
    KINTERRUPT_MODE InterruptMode;
    KINTERRUPT_POLARITY Polarity;
    PULONG PolarityValue;
    NTSTATUS Status;
    LARGE_INTEGER Timeout;
    ULONG Vector;
    KDPC WatchdogDpc;
    KTIMER WatchdogTimer;

    WaitForDebounceTimerDpcs = FALSE;

    //
    // While enabling interrupts for ACPI eventing, mark the GPIO controller as
    // not stoppable or removable. This is because IRP are sent to self IO
    // target and those could be blocked if stop or removal happens in
    // parallel.
    //

    if (Enable != FALSE) {
        WdfDeviceSetStaticStopRemove(GpioExtension->Device, FALSE);
        if (InterruptAction == AcpiInterruptDeepMaskUnmask) {
            GpiopPrepareForAcpiEvents(GpioExtension);
        }
    }

    if (BugCheckAfterSeconds != 0) {
        InterlockedExchange(&GpioExtension->EventData.EnableDisableVector,
                            -1);

        InterlockedExchange(&GpioExtension->EventData.EnableDisableStage,
                            ACPI_ENABLE_DISABLE_STAGE_PINS);

        KeInitializeTimer(&WatchdogTimer);
        KeInitializeDpc(&WatchdogDpc,
                        GpiopEnableDisableAcpiEventsWatchdog,
                        (PVOID)GpioExtension);

        Timeout.QuadPart = Int32x32To64(-10000000, BugCheckAfterSeconds);
        KeSetTimer(&WatchdogTimer, Timeout, &WatchdogDpc);
    }

    //
    // Walk over all the GPIO interrupt descriptors and connect each one of
    // them. Ignore non-interrupt descriptors to allow as many events to be
    // enabled for eventing as possible.
    //

    DescriptorList = GpioExtension->EventData.DescriptorList;
    for (Count = 0; Count < DescriptorList->Count; Count += 1) {
        Descriptor = &DescriptorList->Descriptors[Count];
        if (Descriptor->Type != CmResourceTypeInterrupt) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_ACPIEVT,
                        "%s: Found a non-GPIO interrupt descriptor described"
                        "under _AEI method! Index = %d, Type = %d. Skipping.\n",
                        __FUNCTION__,
                        Count,
                        Descriptor->Type);

            continue;
        }

        if ((Descriptor->Flags & CM_RESOURCE_INTERRUPT_LEVEL_LATCHED_BITS) ==
            CM_RESOURCE_INTERRUPT_LEVEL_SENSITIVE) {

            InterruptMode = LevelSensitive;

        } else {
            InterruptMode = Latched;
        }

        PolarityValue = (PULONG)(&Descriptor->u.Interrupt.MaximumVector) + 1;
        Polarity = VECTOR_FLAGS_TO_INTERRUPT_POLARITY(*PolarityValue);

        GPIO_ASSERT(Polarity != InterruptPolarityUnknown);

        //
        // If events are to be enabled then call the enable interrupt handler,
        // otherwise call the disable interrupt handler.
        //
        // N.B. For enabling events, it is important to synchronize to
        //      those actions with device power state managed by WDF. Hence
        //      call the routine that sends the self-IRPs. When disabling
        //      events, IRPs must not be issued. This because the request
        //      is processed synchronously in the context of the STOP/REMOVE
        //      request and WDF will pend the request until the STOP/REMOVE
        //      is completed, thereby creating a deadlock.
        //
        // If AffectNonWakePinsOnly is specified, enable/disable only non-wake
        // interrupts.
        //

        if ((AffectNonWakePinsOnly == FALSE) ||
            ((Descriptor->Flags & CM_RESOURCE_INTERRUPT_WAKE_HINT) == 0)) {

            Status = STATUS_SUCCESS;
            Vector = Descriptor->u.Interrupt.MinimumVector;
            InterlockedExchange(&GpioExtension->EventData.EnableDisableVector,
                                (LONG)Vector);

            //
            // For all cases (mask, unmask, disable, enable), even though the
            // line needs to be manipulated on *this* controller, the request
            // is still sent via the HAL.
            //
            // This done such that the HAL is aware of the interrupt state for
            // this line (so someone enumerating interrupts sees this) and the
            // GPIO hub has all the mappings built up to support interrupt
            // replay on this line.
            //

            if ((InterruptAction == AcpiInterruptMaskUnmask) ||
                (InterruptAction == AcpiInterruptDeepMaskUnmask)) {

                //
                // The interrupt can only be masked or unmasked if it was
                // successfully enabled.
                //
                
                if (Descriptor->Spare2 == GPIO_ACPI_EVENT_ENABLED) {

                    //
                    // The mask/unmask return value is ignored on failure
                    // because for:
                    //
                    // * Memory-mapped controllers:
                    //       The class extension will force a bugcheck anyway.
                    // 
                    // * Off-soc controllers:
                    //       The class extension will retry several times before
                    //       masking the controller's interrupt on its parent.
                    //

                    if (Enable != FALSE) {
                        (VOID)GpioHalUnmaskInterrupt(
                                    Vector, 
                                    HAL_MASK_UNMASK_FLAGS_NONE);

                    } else {
                        (VOID)GpioHalMaskInterrupt(Vector,
                                                   HAL_MASK_UNMASK_FLAGS_NONE);

                        if (InterruptAction == AcpiInterruptDeepMaskUnmask) {
                            WaitForDebounceTimerDpcs =
                                (WaitForDebounceTimerDpcs ||
                                    GpiopWaitForCompletionOfMaskedEventingPin(
                                        GpioExtension,
                                        Vector));
                        }
                    }
                }

            //
            // Enable/Disable.
            //

            } else {
                if (Enable != FALSE) {

                    //
                    // The interrupt needs to be enabled only if it was
                    // previously disabled (this includes the initial state
                    // of the interrupt).
                    //

                    if (Descriptor->Spare2 == GPIO_ACPI_EVENT_NOT_ENABLED) {
                        Status = GpioHalEnableInterrupt(Vector, InterruptMode, Polarity);
                    }

                } else {

                    //
                    // The interrupt needs to be disabled only if it was
                    // previously enabled.
                    //
                    
                    if (Descriptor->Spare2 == GPIO_ACPI_EVENT_ENABLED) {
                        Status = GpioHalDisableInterrupt(Vector, InterruptMode, Polarity);
                    }
                }

                if (NT_SUCCESS(Status)) {

                    //   
                    // Set the Spare2 field to appropriate GPIO_ACPI_EVENT_*_ENABLED
                    // value depending whether the interrupt was enabled or disabled.
                    //
                    // N.B. The IO descriptor is only used internally within this
                    //      module and thus it is OK to reuse the spare field for state
                    //      tracking.
                    //
                
                    Descriptor->Spare2 = Enable;
                }
            }

            //
            // N.B. If we did nothing, we still log as success.
            //

            if (!NT_SUCCESS(Status)) {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_ACPIEVT,
                            "%s: Failed to %s GPIO interrupt! for ACPI event! "
                            "Vector = %#x, Mode = %d, Polarity = %d, "
                            "Action = %x\n",
                            __FUNCTION__,
                            (Enable != FALSE) ? "enable" : "disable",
                            Vector,
                            InterruptMode,
                            Polarity,
                            (ULONG)InterruptAction);

            } else {
                TraceEvents(GpioExtension->LogHandle,
                            Info,
                            DBG_ACPIEVT,
                            "%s: Successfully %s GPIO interrupt for ACPI event! "
                            "Vector = %#x, Mode = %d, Polarity = %d, "
                            "Action = %x\n",
                            __FUNCTION__,
                            (Enable != FALSE) ? "enabled" : "disabled",
                            Vector,
                            InterruptMode,
                            Polarity,
                            (ULONG)InterruptAction);
            }
        }
    }

    if (BugCheckAfterSeconds != 0) {
        InterlockedExchange(&GpioExtension->EventData.EnableDisableStage,
                            ACPI_ENABLE_DISABLE_STAGE_EVENT_METHODS);
    }

    if (Enable != FALSE) {
        WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);

    } else {
        if (InterruptAction == AcpiInterruptDeepMaskUnmask) {
            if (WaitForDebounceTimerDpcs != FALSE) {
                KeFlushQueuedDpcs();
            }

            GpiopFlushPendingAcpiEvents(GpioExtension);
        }
    }

    if (BugCheckAfterSeconds != 0) {

        //
        // Cancel the watchdog timer. If the DPC has already begun execution, wait
        // for it to finish (the DPC is allocated from this stack).
        //

        if (KeCancelTimer(&WatchdogTimer) == FALSE) {
            KeFlushQueuedDpcs();
        }

        InterlockedExchange(&GpioExtension->EventData.EnableDisableStage,
                            ACPI_ENABLE_DISABLE_STAGE_NONE);

        InterlockedExchange(&GpioExtension->EventData.LastAcpiMethod, 0);
    }

    return STATUS_SUCCESS;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopPrepareForAcpiEvents(
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine prepares the GPIO controller before ACPI eventing pins are
    deep-unmasked after a prior deep-mask (see GpiopFlushPendingAcpiEvents()).

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    None.

--*/

{
    
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_ACPIEVT,
                "%s: Preparing for events again. Extn = 0x%p\n",
                __FUNCTION__,
                GpioExtension);

    ExReInitializeRundownProtection(
        &GpioExtension->EventData.EventWorkerReferences);

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopFlushPendingAcpiEvents (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine waits for all ACPI eventing activity (GpiopProcesssAcpiEventsDpc()
    and GpiopProcessAcpiEventsWorker()) to complete.

    If ACPI eventing pins are to be later deep-unmasked, GpiopPrepareForAcpiEvents()
    must be called before the deep-unmask.

    N.B. This routine assumes that:
    
         1. All ACPI eventing pins have already been disabled or deep-masked.

         2. No other thread is concurrently attempting to enable an ACPI
            eventing pin.  Currently, this is true because the caller is
            GpiopEnableDisableAcpiEvents() and this caller cannot run
            concurrently with itself as it originates from
            GpiopManageAcpiEventing() and that offers synchronization.

         3. It was reached through GpiopManageAcpiEventing() so that it is
            synchronized against AcpiEventStateDestroy.  There's actually no
            dependency on this for now, but it keeps the code consistent.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{
            
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_ACPIEVT,
                "%s: Extn = 0x%p\n",
                __FUNCTION__,
                GpioExtension);

    //
    // The call to the routine originates from GpioClxWdmPreprocessPowerIrp().
    //
    // If there is eventing activity (which occurs on other threads),
    // the following code will wait for such activity to complete:
    // 
    //     [1] GpiopProcessAcpiEvents() ->
    //         [2] GpiopProcessAcpiEventsDpc() ->
    //             [3] GpiopProcessAcpiEventsWorker()
    //
    // This waiting avoids races such as the following: If the current
    // routine were to return, WDF could tear down an on-SoC
    // controller's bank's WDFINTERRUPT object using the current thread
    // while GpiopProcessAcpiEventsWorker() on another thread is still holding
    // the bank interrupt lock corresponding to that WDFINTERRUPT.
    //
    // --
    //
    // At this point, no threads can be in [1] and no new call can
    // be made to [1] because all eventing pins have been disabled or
    // deep-masked.  Since [1] acquires EventWorkerReferences, no more acquires
    // of EventWorkerReferences will occur.
    //
    // Wait for all threads in [3] to complete.
    //

    ExWaitForRundownProtectionRelease(
        &GpioExtension->EventData.EventWorkerReferences);

    ExRundownCompleted(&GpioExtension->EventData.EventWorkerReferences);
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_ACPIEVT,
                "%s: Saw event worker complete. Extn = 0x%p\n",
                __FUNCTION__,
                GpioExtension);

    return STATUS_SUCCESS;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopValidateMethodsForEvents (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_bcount(DescriptorListSize) PIO_RESOURCE_LIST DescriptorList,
    __in ULONG DescriptorListSize
    )

/*++

Routine Description:

    This function validates that the resources specified within the _AEI
    objects are all GPIO interrupt resources and that the pins specified are
    valid.

Arguments:

    DescriptorList - Supplies a pointer to the list of NT IO descriptors that
        describe GPIO pins to be connected.

    DescriptorListSize - Supplies the size of the descriptor list buffer.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG Count;
    PIO_RESOURCE_DESCRIPTOR Descriptor;
    ULONG MinimumSize;
    ULONG RequiredSize;
    NTSTATUS Status;
    BOOLEAN Valid;

    PAGED_CODE();

    //
    // If there are no events to be processed or the buffer is not of
    // adequate length, then return failure. This will cause the state
    // to be marked as failed, which in turn will prevent the state machine
    // from allocating resources or enabling events.
    //

    MinimumSize = FIELD_OFFSET(IO_RESOURCE_LIST, Descriptors);
    Status = STATUS_UNSUCCESSFUL;
    if ((DescriptorListSize < MinimumSize) ||
        (DescriptorList == NULL) ||
        (DescriptorList->Count == 0)) {

        goto ValidateMethodsForEventsEnd;

    } else {
        RequiredSize = MinimumSize +
            (DescriptorList->Count * sizeof(IO_RESOURCE_DESCRIPTOR));

        if (RequiredSize > DescriptorListSize) {
            goto ValidateMethodsForEventsEnd;
        }
    }

    Valid = FALSE;
    for (Count = 0; Count < DescriptorList->Count; Count += 1) {
        Descriptor = &DescriptorList->Descriptors[Count];
        if (Descriptor->Type != CmResourceTypeInterrupt) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_ACPIEVT,
                        "%s: Found a non-GPIO interrupt descriptor described "
                        "under _AEI method! Index = %d, Type = %d. Skipping.\n",
                        __FUNCTION__,
                        Count,
                        Descriptor->Type);

            continue;
        }

        Valid = TRUE;

        //
        // Set the Spare2 field to zero. This field is used to determine if
        // the interrupt was enabled successfully or not.
        //

        Descriptor->Spare2 = GPIO_ACPI_EVENT_NOT_ENABLED;
    }

    if (Valid != FALSE) {
        Status = STATUS_SUCCESS;
    }

ValidateMethodsForEventsEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopValidateEventMethodForPin (
    __in PDEVICE_EXTENSION GpioExtension,
    __in KINTERRUPT_MODE InterruptMode,
    __in PIN_NUMBER PinNumber,
    __out PBOOLEAN RunEVTMethod
    )

/*++

Routine Description:

    This function validates that the pin specified within the _AEI object
    has a valid event handler associated with it (either _EVT or _Exx/_Lxx).

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    InterruptMode - Supplies the trigger mode (edge or level) associated with
        this interrupt.

    PinNumber - Supplies the pin number on the controller that should be
        enabled for interrupt.

    RunEVTMethod - Supplies a pointer that receives whether _EVT should be
        run or not (i.e. _Exx/_Lxx should be run).

Return Value:

    STATUS_SUCCESS if a event handler exists; STATUS_UNSUCCESSFUL otherwise.

--*/

{

    PULONG EdgeLevelPins;
    ULONG EndIndex;
    PGPIO_ACPI_EVENT_DATA EventData;
    BOOLEAN Found;
    ULONG Index;
    ULONG StartIndex;
    NTSTATUS Status;

    PAGED_CODE();

    EventData = &GpioExtension->EventData;
    EdgeLevelPins = EventData->EdgeLevelPins;
    if (InterruptMode == Latched) {
        StartIndex = 0;
        EndIndex =
            EventData->NamespaceEdgePinCount + EventData->ELEEdgePinCount;

    } else {
        StartIndex =
            EventData->NamespaceEdgePinCount + EventData->ELEEdgePinCount;

        EndIndex = StartIndex +
                   EventData->NamespaceLevelPinCount +
                   EventData->ELELevelPinCount;
    }

    //
    // Walk through the pin list and see if there is a match.
    //

    Found = FALSE;
    for (Index = StartIndex; Index < EndIndex; Index += 1) {
        if (PinNumber == EdgeLevelPins[Index]) {
            Found = TRUE;
            break;
        }
    }


    if (Found != FALSE) {
        *RunEVTMethod = FALSE;
        if ((InterruptMode == Latched) &&
            ((Index - StartIndex) >= EventData->NamespaceEdgePinCount)) {

            *RunEVTMethod = TRUE;

        } else {
            if ((InterruptMode == LevelSensitive) &&
                ((Index - StartIndex) >= EventData->NamespaceLevelPinCount)) {

                *RunEVTMethod = TRUE;
            }
        }


    } else if ((EventData->EVTMethodDefined != FALSE) &&
               (EventData->ELEObjectDefined == FALSE)) {

        Found = TRUE;
        *RunEVTMethod = TRUE;
    }

    if (Found == FALSE) {
        Status = STATUS_UNSUCCESSFUL;

    } else {
        Status = STATUS_SUCCESS;
    }

    return Status;
}

VOID
GpiopProcessAcpiEvents (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine is responsible for getting pending ACPI events processed on
    the GPIO controller. It is responsible for queuing the event DPC if it
    is not already running.

    N.B. This routine can be called from DIRQL (for memory-mapped GPIOs)
         and PASSIVE IRQL (for serially-accessible GPIOs). This routine assumes
         caller has held the appropriate lock before calling in. The interrupt
         lock must be held if called from DIRQL; the controller lock must be
         held if called from PASSIVE.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    None.

--*/

{

    BOOLEAN AcquiredReference;
    KIRQL OldIrql;
    BOOLEAN QueueDpc;
    LONG PreviousState;
    PLONG WorkerRunning;

    //
    // The sole purpose of the event DPC is to get the worker queued. If the
    // worker is already running, then there is no need to queue the DPC.
    // The worker will pick up the new ACPI events and process it.
    //
    // If the worker is not running and neither is the DPC, then queue the
    // DPC.
    //

    QueueDpc = FALSE;
    OldIrql = KeGetCurrentIrql();
    WorkerRunning = &GpioExtension->EventData.EventWorkerRunning;
    PreviousState = InterlockedCompareExchange(WorkerRunning,
                                               GPIO_WORKER_RUNNING,
                                               GPIO_WORKER_NOT_RUNNING);

    //
    // If the worker is not already running, then schedule it to run.
    //
    // If this routine is called from within the interrupt context, then
    // queue a DPC to run. Otherwise, invoke it synchronously from the current
    // thread.
    //

    if (PreviousState == GPIO_WORKER_NOT_RUNNING) {

        //
        // Add an outstanding reference to the worker.
        //
        // After ACPI event pins are deep-masked, and before they are
        // deep-unmasked, EventWorkerReferences is in a run-down state.
        // Inside that window, if the current routine were called and the
        // below acquire were attempted, the acquire would fail.  However,
        // inside that window, the pins are deep-masked so the current routine
        // wont't be called.
        //
        // N.B. ExAcquireRundownProtection() is potentially being called here
        //      at DIRQL.  Don't add a call here to ExReleaseRundownProtection()
        //      since the latter requires <= DISPATCH_LEVEL.
        //

        AcquiredReference =
            ExAcquireRundownProtection(
                   &GpioExtension->EventData.EventWorkerReferences);

#if defined(DBG)

        if (AcquiredReference == FALSE) {
            __debugbreak();
        }

#else

        UNREFERENCED_PARAMETER(AcquiredReference);

#endif

        if (OldIrql >= DISPATCH_LEVEL) {
            WdfDpcEnqueue(GpioExtension->EventData.EventDpc);

        } else {
            GpiopProcessAcpiEventsCommon(GpioExtension->EventData.EventDpc);
        }
    }

    return;
}

_Use_decl_annotations_ 
VOID
GpiopProcessAcpiEventsDpc (
    WDFDPC Dpc
    )

/*++

Routine Description:

    This routine is the DPC callback for ACPI event processing.

Arguments:

    Dpc - Supplies a handle to the WDF DPC object.

Return Value:

    None.

--*/

{

    GpiopProcessAcpiEventsCommon(Dpc);
    return;
}

_Use_decl_annotations_
VOID
GpiopProcessAcpiEventsCommon(
    WDFDPC Dpc
    )

/*++

Routine Description:

    This routine is the callback for ACPI event procesing. This routine is
    responsible for queuing a work item to process the interrupts for this GPIO
    controller.

    Due to the use of GpioExtension->EventData.EventWorkerRunning, only one
    instance of this callback can be active at a time i.e. it is never run in
    parallel with itself.

    N.B. This routine can entered at DISPATCH_LEVEL (for memory-mapped GPIOs) or
         PASSIVE IRQL (for serially-accessible GPIOs). If the call occurs
         at PASSIVE IRQL, then the caller is assumed to have acquired the
         controller lock prior to calling in.

Arguments:

    Dpc - Supplies a handle to the WDF DPC object.

Return Value:

    None.

--*/

{

    BOOLEAN Enqueued;
    PGPIO_ACPI_EVENT_DATA EventsData;
    PDEVICE_EXTENSION GpioExtension;

    //
    // Enqueue the workitem to the system's work queue (to get it running).
    // The DPC is only ever scheduled if the work item is not already running.
    //

    GpioExtension = GpioClxGetDeviceExtension(WdfDpcGetParentObject(Dpc));

    //
    // Take a reference on the device while it is sitting on the queue.
    // The reference will be dropped at the end of the worker.
    //

    WdfObjectReferenceWithTag(GpioExtension->Device,
                              (PVOID)GPIO_CLX_ACPI_EVENT_POOL_TAG);

    EventsData = &GpioExtension->EventData;
    if (GpioForceEmergencyWorker == FALSE) {
        Enqueued = GpiopExTryQueueWorkItem(&EventsData->EventWdmWorkItem,
                                           DelayedWorkQueue);

    } else {
        Enqueued = FALSE;
    }

    if (Enqueued == FALSE) {
        GpioHubQueueEmergencyWorkItem(&EventsData->EventWdmWorkItem);
    }

    return;
}

_Use_decl_annotations_
VOID
GpiopProcessAcpiEventsWorker (
    PVOID Context
    )

/*++

Routine Description:

    This routine is the worker routine for processes ACPI events that are
    currently pending on the controller. For each of the pending events, it
    will issue a request to ACPI driver (must on the stack) to evaluate the
    appropriate _Lxx or _Exx method based on the interrupt polarity. For
    level-triggered GPIO pins, this routine will unmask the pins once the
    _Lxx method evaluation completes.

Arguments:

    Context -  Supplies a pointer to the GPIO device extension.

Return Value:

    None.

--*/

{

    BANK_ID BankId;
    PDEVICE_EXTENSION GpioExtension;
    ACPI_EVAL_INPUT_BUFFER_SIMPLE_INTEGER InputBuffer;
    ULONG InputBufferLength;
    WDF_MEMORY_DESCRIPTOR InputDescriptor;
    KINTERRUPT_MODE InterruptMode;
    CHAR InterruptModeToChar[] = {'L', 'E'};
    WDFIOTARGET IoTarget;
    union {
        CHAR AsChar[5];
        ULONG AsUlong;
    } MethodName;

    PIN_NUMBER NextEventPin;
    KIRQL OldIrql;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER RelativePin;
    USHORT RetryCount;
    ULONG_PTR SizeReturned;
    NTSTATUS Status;
    ULONG Virq;
    PLONG WorkerRunning;

    GpioExtension = (PDEVICE_EXTENSION)Context;
    IoTarget = WdfDeviceGetIoTarget(GpioExtension->Device);
    RtlZeroMemory(&MethodName, sizeof(MethodName));
    NextEventPin = 0;

    GPIO_ACQUIRE_ACPI_EVENT_LOCK(GpioExtension, &OldIrql);

    GPIO_ASSERT(OldIrql == PASSIVE_LEVEL);

    WHILE(TRUE) {

        //
        // Find the next event pin number. If the current of set of events
        // have all been serviced, then exit the loop.
        //

        NextEventPin = GpiopFindNextEventPinAndClear(GpioExtension,
                                                     NextEventPin);

        if (NextEventPin == INVALID_PIN_NUMBER) {
            break;
        }

        //
        // Drop the interrupt and controller locks prior to running the method.
        //

        GPIO_RELEASE_ACPI_EVENT_LOCK(GpioExtension, OldIrql);

        //
        // N.B. Since the pin is connected for ACPI eventing, the pin
        //      information (including mode and VIRQ) cannot change until ACPI
        //      eventing is destroyed. WDF will delete the device only after
        //      all the workers have finished.
        //

        PinInformation = GpiopGetPinEntryFromPinNumber(GpioExtension,
                                                       NextEventPin);

        GPIO_ASSERT(PinInformation != NULL);

        Virq = PinInformation->Virq;
        InterruptMode = PinInformation->InterruptMode;
        if (PinInformation->Mode.RunEVTMethod != FALSE) {
            MethodName.AsUlong = GPIO_LONG_PIN_EVENT_METHOD_AS_ULONG;
            MethodName.AsChar[4] = '\0';

        } else {
            Status = RtlStringCbPrintfA(&MethodName.AsChar[0],
                                        sizeof(MethodName.AsChar),
                                        "_%c%02X",
                                        InterruptModeToChar[InterruptMode],
                                        NextEventPin);

            GPIO_ASSERT(NT_SUCCESS(Status) != FALSE);
        }

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_ACPIEVT,
                    "%s: EVENT -> Executing method %s [Pin = %#x]...\n",
                    __FUNCTION__,
                    &MethodName.AsChar[0],
                    NextEventPin);

        if (PinInformation->Mode.RunEVTMethod != FALSE) {
            InputBuffer.Signature =
                ACPI_EVAL_INPUT_BUFFER_SIMPLE_INTEGER_SIGNATURE;

            InputBuffer.IntegerArgument = NextEventPin;
            InputBufferLength = sizeof(ACPI_EVAL_INPUT_BUFFER_SIMPLE_INTEGER);

        } else {
            InputBuffer.Signature = ACPI_EVAL_INPUT_BUFFER_SIGNATURE;
            InputBufferLength = sizeof(ACPI_EVAL_INPUT_BUFFER);
        }

        InputBuffer.MethodNameAsUlong = *(PULONG)&MethodName.AsUlong;
        WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&InputDescriptor,
                                          (PVOID)&InputBuffer,
                                          InputBufferLength);

        InterlockedExchange(&GpioExtension->EventData.LastAcpiMethod,
                            (LONG)InputBuffer.MethodNameAsUlong);

        //
        // Run the appropriate _Lxx or _Exx ACPI method. Since running the
        // method may be crucial for certain scenarios, attempt a retry (few
        // times) if method execution fails for some reason (e.g. lack of
        // resources).
        //

        RetryCount = 0;
        do {
            BankId = GPIO_BANK_ID_FROM_PIN_NUMBER(GpioExtension, NextEventPin);
            RelativePin = GPIO_ABSOLUTE_TO_RELATIVE_PIN(GpioExtension,
                                                        NextEventPin);
            EventWrite_ACPI_EVENT_METHOD_START(
                &GpioExtension->Banks[BankId].ActivityId,
                GpioExtension->BiosName.Buffer,
                BankId,
                RelativePin);

            Status = WdfIoTargetSendIoctlSynchronously(IoTarget,
                                                       NULL,
                                                       IOCTL_ACPI_EVAL_METHOD,
                                                       &InputDescriptor,
                                                       NULL,
                                                       NULL,
                                                       &SizeReturned);
        
            EventWrite_ACPI_EVENT_METHOD_COMPLETE(
                &GpioExtension->Banks[BankId].ActivityId,
                RelativePin);

            if ((Status == STATUS_OBJECT_NAME_NOT_FOUND) ||
                (Status == STATUS_NO_SUCH_DEVICE)) {

                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_ACPIEVT,
                            "%s: EVENT -> Method %s does not exist! This is "
                            "firmware bug! Status = %#x\n",
                            __FUNCTION__,
                            &MethodName.AsChar[0],
                            Status);

                break;

            } else if (!NT_SUCCESS(Status)) {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_ACPIEVT,
                            "%s: EVENT -> Method %s evaluation failed! "
                            "Status = %#x \n",
                            __FUNCTION__,
                            &MethodName.AsChar[0],
                            Status);

                __fallthrough;      // for retry

            } else {
                TraceEvents(GpioExtension->LogHandle,
                            Info,
                            DBG_ACPIEVT,
                            "%s: EVENT -> Method %s evaluation succeeded!\n",
                            __FUNCTION__,
                            &MethodName.AsChar[0]);

                break;
            }

            RetryCount += 1;
        } while(RetryCount < GPIO_ACPI_EVENT_MAXIMUM_RETRIES);

        //
        // Unmask the event pin if it is level-triggered. If the _Lxx method
        // failed to execute, then the whole process will repeat again.
        // If there is a platform and/or ACPI FW issue, this could result in
        // an interrupt storm.
        //
        //  N.B. The unmask request is not routed through the HAL as the
        //       corresponding mask requested never was. Refer to comment in
        //       interrupt service routine on why the mask request was not
        //       routed.
        //

        if (InterruptMode == LevelSensitive) {
            GpioClxUnmaskInterrupt(GpioExtension,
                                   HAL_MASK_UNMASK_FLAGS_SERVICING_COMPLETE,
                                   Virq);
        }

        GPIO_ACQUIRE_ACPI_EVENT_LOCK(GpioExtension, &OldIrql);
    }

    //
    // Mark the worker as done running.
    //
    // After this write, another instance of the worker can be scheduled and
    // start running (in parallel).
    //
    // N.B. This write must be performed before GPIO_ACPI_EVENT_DATA::PendingEventsBitmap
    //      is allowed to change.  That's because the current instance of the
    //      worker isn't going to be processing more pins.
    //

    WorkerRunning = &GpioExtension->EventData.EventWorkerRunning;
    InterlockedExchange(WorkerRunning, GPIO_WORKER_NOT_RUNNING);

    //
    // Currently, PendingEventsBitmap is empty.
    // As soon as this lock is released, that can change.
    //
    
    GPIO_RELEASE_ACPI_EVENT_LOCK(GpioExtension, OldIrql);

    GPIO_ASSERT(OldIrql == PASSIVE_LEVEL);

    //
    // After this release, don't touch anything related to WDFDEVICE resources
    // (e.g. the bank interrupt lock, which can include GPIO_ACQUIRE_ACPI_EVENT_LOCK())
    // in case GpiopFlushPendingAcpiEvents() had been waiting for the release
    // and intends to return control to WDF to tear down resources.
    //
 
    ExReleaseRundownProtection(&GpioExtension->EventData.EventWorkerReferences);

    //
    // Drop the reference to the device which was taken when it was queued.
    //

    WdfObjectDereferenceWithTag(GpioExtension->Device,
                                (PVOID)GPIO_CLX_ACPI_EVENT_POOL_TAG);

    return;
}

PIN_NUMBER
GpiopFindNextEventPinAndClear (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_opt PIN_NUMBER Hint
    )

/*++

Routine Description:

    This routine examines the GPIO controller's pending ACPI event register
    and determines the next event pin number.

    N.B. This routine will clear the return bit number in the buffer on exit.
         Assumes the interrupt lock is held by the caller.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Hint - Supplies a hint on where to start the search for next pin.

Return Value:

    PIN_NUMBER - The interrupting pin's number. INVALID_PIN_NUMBER if no pin
        is marked as interrupting.

--*/

{

    PGPIO_ACPI_EVENT_DATA EventData;
    PIN_NUMBER PinNumber;
    ULONG Value;

    //
    // Find the first set bit in the status register that is not currently
    // masked.
    //

    EventData = &GpioExtension->EventData;
    Value = RtlFindSetBitsAndClear(&EventData->PendingEventsBitmap, 1, Hint);
    if (Value != MAX_ULONG) {
        PinNumber = (PIN_NUMBER)Value;

    } else {
        PinNumber = INVALID_PIN_NUMBER;
    }

    return PinNumber;
}

BOOLEAN
GpiopIsAcpiEventInterrupt (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Gsiv
    )

/*++

Routine Description:

    This routine determines if the supplied GSIV maps to an ACPI event pin or
    not.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Gsiv - Supplies the GSIV value to be checked.

Return Value:

    TRUE if the GSIV maps to an ACPI event pin; FALSE otherwise.

--*/

{

    BOOLEAN AcpiEvent;
    ULONG Count;
    PIO_RESOURCE_DESCRIPTOR Descriptor;
    PIO_RESOURCE_LIST DescriptorList;

    PAGED_CODE();

    //
    // Walk over all the GPIO interrupt descriptors to determine if the supplied
    // GSIV matches any ACPI event descriptor.
    //

    AcpiEvent = FALSE;
    DescriptorList = GpioExtension->EventData.DescriptorList;
    if (DescriptorList == NULL) {
        goto IsAcpiEventInterruptEnd;
    }

    for (Count = 0; Count < DescriptorList->Count; Count += 1) {
        Descriptor = &DescriptorList->Descriptors[Count];
        if (Descriptor->Type == CmResourceTypeInterrupt) {
            if (Gsiv == Descriptor->u.Interrupt.MinimumVector) {
                AcpiEvent = TRUE;
                break;
            }
        }
    }

IsAcpiEventInterruptEnd:
    return AcpiEvent;
}

_Function_class_(POWER_SETTING_CALLBACK)
_IRQL_requires_same_
NTSTATUS
GpiopPowerAcpiEventingCallback (
    _In_ LPCGUID SettingGuid,
    _In_reads_bytes_(ValueLength) PVOID Value,
    _In_ ULONG ValueLength,
    _Inout_opt_ PVOID Context
    )

/*++

Routine Description:

    This routine handles GUID_PDC_IDLE_RESILIENCY_ENGAGED notification from PDC
    to disable or enable non-wake AEIs in order to prepare the platform for
    DRIPS transition.

Arguments:

    SettingGuid - Supplies a pointer to GUID which this invocation is for.
        When dereferenced, it is guaranteed to be equal to GUID_PDC_IDLE_
        RESILIENCY_ENGAGED DISABLED.

    Value - Supplies a pointer to a ULONG indicating whether we are preparing
        for DRIPS transition or exiting from it.

    ValueLength - Supplies the length of the buffer pointed by Value. It is
        guaranteed to be sizeof(ULONG).

    Context - Supplies the PVOID value when PoRegisterPowerSettingCallback is
        called. This is a pointer to the DEVICE_EXTENSION of a GPIO controller.

Return Value:

    NTSTATUS code.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    ULONG NonWakeEventingDisabled;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(SettingGuid);

    GPIO_ASSERT(IsEqualGUID(SettingGuid, &GUID_PDC_IDLE_RESILIENCY_ENGAGED));

    GpioExtension = (PDEVICE_EXTENSION)Context;

    GPIO_CLX_VALIDATE_SIGNATURE(GpioExtension);

    if (ValueLength != sizeof(ULONG)) {
        return STATUS_INVALID_PARAMETER;
    }

    NonWakeEventingDisabled = *(PULONG)Value;
    if (GpioExtension->EventData.NonWakeEventingDisabled !=
        NonWakeEventingDisabled) {

        GpioExtension->EventData.NonWakeEventingDisabled =
            NonWakeEventingDisabled;

        if (NonWakeEventingDisabled == 0) {
            Status = GpiopManageAcpiEventing(
                GpioExtension,
                AcpiEventStateEnableForConnectedStandby,
                TRUE);

        } else {
            Status = GpiopManageAcpiEventing(
                GpioExtension,
                AcpiEventStateDisableForConnectedStandby,
                TRUE);
        }

    } else {
        Status = STATUS_SUCCESS;
    }

    return Status;
}

// ----------------------------------------------------------- Locking routines

VOID
GpiopAcquireAcpiEventLock (
    __in PDEVICE_EXTENSION GpioExtension,
    __out __drv_out_deref(__drv_savesIRQL) PKIRQL OldIrql
    )

/*++

Routine Description:

    This routine acquires the appropriate ACPI event lock. For memory-mapped
    GPIO controllers this translates to the event interrupt lock. For
    serially-accessible GPIO controllers, this translates to the passive-level
    event lock.

Arguments:

    OldIrql - Supplies a pointer to a variable that receives the IRQL on entry.

Return Value:

    None.

--*/

{

    PGPIO_ACPI_EVENT_DATA EventData;

    EventData = &GpioExtension->EventData;
    if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) {
        KeRaiseIrql(GpioExtension->SynchronizationIrql, OldIrql);
        KeAcquireSpinLockAtDpcLevel(&EventData->EventInterruptLock);

    } else {
        *OldIrql = KeGetCurrentIrql();
        ExAcquireFastMutex(&EventData->EventLock);
    }

    return;
}


__drv_setsIRQL(OldIrql)
FORCEINLINE
VOID
GpiopReleaseAcpiEventLock (
    __in PDEVICE_EXTENSION GpioExtension,
    __in __drv_in(__drv_restoresIRQL) KIRQL OldIrql
    )

/*++

Routine Description:

    This routine releases the appropriate ACPI event lock.

Arguments:

    OldIrql - Supplies the original IRQL value.

Return Value:

    None.

--*/

{

    PGPIO_ACPI_EVENT_DATA EventData;

    EventData = &GpioExtension->EventData;
    if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) {
        KeReleaseSpinLockFromDpcLevel(&EventData->EventInterruptLock);
        KeLowerIrql(OldIrql);

    } else {
        ExReleaseFastMutex(&EventData->EventLock);
    }

    GPIO_ASSERT(KeGetCurrentIrql() == OldIrql);

    return;
}

